const { dbQuery } = require("../../../../config/db.config");
const { postUrlExist, urlExist } = require("../../../../utils/url_exist");
const { errorMsgEmpty } = require("../../../../utils/errors_message");
const {
  SC_ASC_ARTICLES,
  SC_ARTICLES_CATEGORY,
} = require("../../../../models/tables");
const {
  articleFields,
  articleMandatoryFields,
} = require("./create_article_content");
const {
  getArticlesService,
  getArticlesWithPaginationService,
  deleteArticleService,
  updateArticlesService,
  addArticlesService,
} = require("../../../../services/article.service");
const { areAllKeysPresent } = require("../../../../utils/common_utils");
const tryCatch = require("../../../../utils/tryCatch");
const updateColumn = require("../../../../utils/updateColumn");

const table = SC_ASC_ARTICLES;
const categoryTbl = SC_ARTICLES_CATEGORY;

// @desc get articles
// @api GET api/v1/admin/articles
// @access private
const get_articles = tryCatch(async (req, res) => {
  const { search = "", page = 1, limit = 30, id } = req.query;


  

  if (Number(page) && Number(page) < 1) throw new Error("Invalid page number");
  const offset = (Number(page) - 1) * Number(limit);

  
  const queryMap = {
    attributes: [
      `${table}.nr_title`,
      `${table}.nr_active`,
      `${table}.nr_publish_date`,
      `${table}.nr_update_on`,
      `${table}.nr_priority`,
      `${table}.nr_package_bottom_status`,
      `${categoryTbl}.nc_name`,
      `${table}.nc_id`,
      `${table}.nr_id`,
    ],
    joinTbl: [
      `LEFT JOIN ${categoryTbl} ON ${categoryTbl}.nc_id = ${table}.nc_id`,
    ],
    
    additional: [`order by nr_id desc LIMIT ${limit} OFFSET ${offset}`],
    conditions: [],
    parameters: [],
  };

  if (search) {
    queryMap.conditions.push(`nr_title LIKE ?`);
    queryMap.parameters.push(`%${search}%`);
  }
  if (id) {
    queryMap.attributes = [`${table}.*`];
    queryMap.conditions.push(`${table}.nr_id = ?`);
    queryMap.parameters.push(id);
    queryMap.additional = [`LIMIT 1`];
    
    

    const result = await getArticlesService(queryMap);
    if (result instanceof Error) errorMsgEmpty("No data found", res);

    return res.status(200).json({ success: true, data: result.pop() });
  }

  
  const { result, total } = await getArticlesWithPaginationService(queryMap);
  if (result instanceof Error) errorMsgEmpty("No data found", res);

  return res.status(200).json({ success: true, data: result, records: total });
});

// @desc post articles
// @api POST api/v1/admin/articles
// @access private ADMIN
const post_articles = tryCatch(async (req, res) => {
  const requiredFields = [
    "nc_id",
    "nr_title",
    "nr_front_url",
    "nr_image",
    "nr_big_image",
    "nr_description",
    "nr_large_description",
    "nr_priority",
    "nr_active",
    "nr_mtitle",
    "nr_mkeyword",
    "nr_mdescription",
  ];
  if (!areAllKeysPresent(req.body, requiredFields)) {
    throw new Error("All fields are required");
  }

  req.body.nr_front_url &&
    (await postUrlExist(table, "nr_front_url", req.body.nr_front_url));

  const queryMap = {
    attributes: [
      ...requiredFields,
      "nr_publish_date",
      "nr_date",
      "posted_by",
      "page_id",
    ],
    parameters: [
      ...requiredFields.map((field) => req.body[field]),
      new Date(),
      new Date(),
      "",
      0,
    ],
  };

  const result = await addArticlesService(queryMap);
  if (result instanceof Error) {
    res.statusCode = 400;
    throw result;
  }

  return res.status(200).json({
    success: true,
    message: `Article added successfully`,
  });
});

// @desc update articles
// @api PUT api/v1/admin/articles
// @access private ADMIN
const update_articles = tryCatch(async (req, res) => {
  const { nr_id, update } = req.body;
  if (!nr_id || !update) throw new Error("Invalid id or update");

  update?.nr_front_url &&
    (await urlExist(
      table,
      "nr_id",
      "nr_front_url",
      update?.nr_front_url,
      nr_id
    ));

    
  const { column, value } = updateColumn(update);

  value[14]=new Date(); ///// Update nr_update_on field value 

  const queryMap = {
    update: [column, ].join(", "),
    conditions: [`nr_id = ?`],
    parameters: [...value, nr_id],
  };

  const result = await updateArticlesService(queryMap);
  if (result instanceof Error) throw result;

  return res
    .status(200)
    .json({ success: true, message: `Article updated successfully` });
});

// @desc delete articles
// @api DELETE api/v1/admin/articles
// @access private ADMIN
const del_articles = tryCatch(async (req, res) => {
  const { nr_id } = req.body;
  if (!nr_id) throw new Error("Invalid id");

  const queryMap = {
    conditions: [`nr_id = ?`],
    parameters: [nr_id],
  };

  const result = await deleteArticleService(queryMap);
  if (result instanceof Error) throw result;

  return res
    .status(200)
    .json({ success: true, message: `Article deleted successfully` });
});

module.exports = {
  get_articles,
  post_articles,
  update_articles,
  del_articles,
};
