const { dbQuery } = require("../../../../config/db.config");
const tryCatch = require("../../../../utils/tryCatch");
const updateColumn = require("../../../../utils/updateColumn");
const { SC_USER, SCHOOL_DETAIL } = require("../../../../models/tables");
const { BUYER, SCHOOL } = require("../../../../common/constants");
const {
  checkBuyerSchoolExist,
  checkBuyerExist,
  checkLeadAlreadyExist,
  checkSchoolExist,
} = require("../../../../utils/leadAccessCheck");
const { errorMsgEmpty } = require("../../../../utils/errors_message");

const table = SC_USER;
const schoolTbl = SCHOOL_DETAIL;

//@desc get one school partner details
//@route GET /api/v1/admin/become-partner/:id
//@access Private ADMIN
const getSchoolPartnerById = tryCatch(async (req, res) => {
  let [sql , result] = [null, null];
  const { id } = req.params;
  const doesBuyerExist = await checkBuyerExist(id);
  if (!doesBuyerExist) {
    res.statusCode = 404;
    throw new Error("No buyer found with this id");
  }
  const attributes = [
    `${table}.us_id`,
    `${table}.us_fname`,
    `${table}.us_lname`,
    `${table}.email_1`,
    `${table}.us_username`,
    `${table}.subscription_date`,
    `${table}.renewal_date`,
    `${table}.export_data`,
    `${table}.mobile_1`
  ];
  const schoolAttributes = [
    `${schoolTbl}.admission_partner`,
    `${schoolTbl}.school_name`,
    `${schoolTbl}.asc_reg_fee`,
    `${schoolTbl}.buyer_name`,
    `${schoolTbl}.sid`,
  ];
  const schoolSql = `SELECT ${schoolAttributes} FROM ${schoolTbl} WHERE buyer_name = ?`;
  const schoolResult = await dbQuery(schoolSql, [id]);
  sql = `SELECT ${attributes.join(", ")} FROM ${table} WHERE 1=1 AND user_type = ? AND  ${table}.us_id = ? LIMIT 1`;
  result = await dbQuery(sql, [BUYER, id]);
  if (!result.length) {
    throw new Error("No school partner found");
  }
  result[0].owner_of = schoolResult;
  res.status(200).json({
    status: "success",
    data: result[0],
  });
});

// @desc GET ALL THE BUYERS
// @route GET /api/v1/admin/become-partner
// @access Private ADMIN
const getAllBuyers = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];

  const attributes = [
    `${table}.us_id`,
    `${table}.us_fname`,
    `${table}.us_lname`,
    `${table}.email_1`,
    `${table}.us_username`,
    `school_detail.school_name`
  ];

  sql = `SELECT ${attributes.join(", ")} FROM ${table} 
  JOIN school_detail ON sc_user.us_id=school_detail.buyer_name
  WHERE 1=1 AND user_type = ? ORDER BY ${table}.us_id DESC`;


  result = await dbQuery(sql, [BUYER]);
  if (!result.length) {
    errorMsgEmpty("No buyers found", res);
  }
  return res.status(200).json({
    status: "success",
    data: result,
  });
});

// @desc ADD school partner
// @route POST /api/v1/admin/become-partner
// @access Private ADMIN
const addSchoolPartner = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const {
    sid, // School ID
    us_fname,
    us_lname,
    email_1,
    us_password,
    subscription_date,
    renewal_date,
    asc_reg_fee, // Registration Fees
    export_data, // Y or N
    admission_partner, // Y or N
    mobile_1,
  } = req.body;
  if (
    !sid ||
    !us_fname ||
    !us_lname ||
    !email_1 ||
    !us_password ||
    !subscription_date ||
    !renewal_date ||
    !asc_reg_fee ||
    !export_data ||
    !admission_partner ||
    !mobile_1
  ) {
    res.statusCode = 400;
    throw new Error("Please fill all the fields");
  }

  const doesSchoolExist = await checkSchoolExist(sid);
  if(!doesSchoolExist) throw new Error("School doesn't exist")
  const doesAnotherBuyerSchoolExist = await checkBuyerSchoolExist(sid, null);
  if(!doesAnotherBuyerSchoolExist){
    throw new Error("This school is already linked, or might not exist")
  }

  
  // Check if email already exists
  if (await checkLeadAlreadyExist(email_1,BUYER)) {
    res.statusCode = 400;
    throw new Error("Email already exists, Please login");
  }

  sql = `INSERT INTO ${table} (us_fname, us_lname, us_username, email_1, us_password, subscription_date, renewal_date, export_data, user_type, mobile_1, us_create_date) VALUES (?,?,?,?,?,?,?,?,?,?,NOW())`;
  result = await dbQuery(sql, [
    us_fname,
    us_lname,
    email_1,
    email_1,
    us_password,
    subscription_date,
    renewal_date,
    export_data, // Y or N
    BUYER,
    mobile_1,
  ]);
  if (!result.affectedRows) {
    throw new Error("Some Error occured while becoming a partner");
  }

  const buyerId = result.insertId;

  const schoolSql = `UPDATE ${schoolTbl} SET buyer_name = ?, asc_reg_fee = ?, admission_partner = ? WHERE sid = ?`;
  const schoolResult = await dbQuery(schoolSql, [
    buyerId,
    asc_reg_fee,
    admission_partner,
    sid,
  ]);

  if (!schoolResult.affectedRows) {
    const t = `DELETE FROM ${table} WHERE us_id = ?`;
    const transaction = await dbQuery(t, [buyerId]);
    if (!transaction.affectedRows) {
      throw new Error(
        "Some Error occured while becoming a partner, buyer is missing"
      );
    }
    throw new Error(
      "Error occured while inserting school info, please re-fill all details"
    );
  }
  return res.status(200).json({
    success: true,
    message: "School Partner added successfully",
  });
});

// @desc if school partner is already exist and want new school to be added
// @route PUT /api/v1/admin/become-partner-add
// @access Private ADMIN
const addExistingBuyerToSchoolPartner = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const {
    sid, // School ID
    us_id, // Buyer ID
    asc_reg_fee, // Registration Fees
  } = req.body;

  if (!sid || !us_id || !asc_reg_fee) {
    res.statusCode = 400;
    throw new Error("Please fill all the fields");
  }

  const doesBuyerExist = await checkBuyerExist(us_id);
  if(!doesBuyerExist) throw new Error("Buyer doesn't exist")
  const doesAnotherBuyerSchoolExist = await checkBuyerSchoolExist(sid, us_id);
  if(!doesAnotherBuyerSchoolExist){
    throw new Error("A different buyer already exist, or might not exist")
  }

  sql = `UPDATE ${schoolTbl} SET buyer_name = ?, asc_reg_fee = ? WHERE sid = ?`;
  result = await dbQuery(sql, [us_id, asc_reg_fee, sid]);

  if (!result.affectedRows) {
    throw new Error(
      "Error occured while inserting school info, please re-fill all details"
    );
  }

  return res.status(200).json({
    success: true,
    message: "New School Partner added successfully to the buyer",
  });
});

//@desc UPDATE school partner details
//@route PUT /api/v1/admin/become-partner
//@access Private ADMIN
const updateSchoolPartner = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { us_id, update } = req.body;
  if (!us_id || !update) throw new Error("All fields are required");
  let schoolLinked = null;
  if (update["owner_of"].length) {
    schoolLinked = update.owner_of;
  }
  delete update["owner_of"];
  const { column, value } = updateColumn(update);
  sql = `UPDATE ${table} SET ${column}, us_update_date = NOW() WHERE us_id = ?`;
  result = await dbQuery(sql, [...value, us_id]);
  if (!result.affectedRows) throw new Error("Error while update buyer details");

  if (schoolLinked) {
    for (item of schoolLinked) {
      if (item.buyer_name != null && us_id != item.buyer_name) {
        throw new Error("Buyer is not the same");
      }
      const schoolId = item.sid;
      if (schoolId == null || schoolId === "") {
        throw new Error("No school Id found");
      }
      const doesSchoolBelongsToBuyer = await checkBuyerSchoolExist(schoolId,us_id);
      if (!doesSchoolBelongsToBuyer) {
        throw new Error("School doesn't belong to you");
      }
      delete item.sid;
      const { column: schoolCol, value: schoolVal } = updateColumn(item);
      const schoolSql = `UPDATE ${schoolTbl} SET ${schoolCol}, update_date = NOW() WHERE sid = ?`;
      const schoolResult = await dbQuery(schoolSql, [...schoolVal, schoolId]);
      if (!schoolResult.affectedRows) {
        throw new Error("Error while updating school info of Buyer");
      }
    }
  }
  return res.status(200).json({
    success: true,
    message: "Successfully updated buyer details",
  });
});

module.exports = {
  addSchoolPartner,
  getAllBuyers,
  addExistingBuyerToSchoolPartner,
  updateSchoolPartner,
  getSchoolPartnerById
};
