const { dbQuery } = require("../../../../../config/db.config");
const tryCatch = require("../../../../../utils/tryCatch");
const { errorMsgEmpty } = require("../../../../../utils/errors_message");
const {
  TBL_FREELY_ENQUIRY,
  COURSE,
  EMAIL_TEMPLATE,
  CAREER_LIST,
  STATE_TBL,
  CITY_TBL
} = require("../../../../../models/tables");
const {
  checkCollegeExist,
  getCollegeBasicDetails,
} = require("../../../../../utils/leadAccessCheck");
const smsTemplate = require("../../EmailTemplate/sms_template_contants");
const axios = require("axios");
const { SMS_USERNAME, SMS_PSWD, SENDER_ID, EMAIL_FROM } = process.env;
const { transporter } = require("../../EmailTemplate/email");
const { getEnquiryPaginationService, deleteEnquiryService, updateEnquiryService, getEnquiryService } = require("../../../../../services/enquiry.service");

let [sql, result] = [null, null];

const table = TBL_FREELY_ENQUIRY;
const courseTbl = COURSE;
const templateTbl = EMAIL_TEMPLATE;

const getSmsDetails = (data, phone) => {
  const smsURL = `http://sms.mishtel.net/api/mt/SendSMS?user=${SMS_USERNAME}&password=${SMS_PSWD}&senderid=${SENDER_ID}&channel=Trans&DCS=0&flashsms=0&number=${phone}&text=${data}(INTASC)&route=1`;
  return smsURL;
};

// @desc get one freely enquiry list detail by id
// @api GET /api/v1/admin/enquiry/career/:id
// @access Private ADMIN
const getFreelyEnquiryDetails = tryCatch(async (req, res) => {

  const { id } = req.params;

  const queryMap = {
    attributes: [`${TBL_FREELY_ENQUIRY}.*`, `${COURSE}.course_name AS course`],
    conditions: [`${TBL_FREELY_ENQUIRY}.id = ?`],
    parameters: [id],
    additional: [`LIMIT 1`],
    joinTbl: [`LEFT JOIN ${COURSE} ON ${COURSE}.course_id = ${TBL_FREELY_ENQUIRY}.course`],
    table: TBL_FREELY_ENQUIRY,
  };

  const result = await getEnquiryService(queryMap);
  if(result instanceof Error || !result[0]){
    res.statusCode = 400;
    throw new Error("Error occured while fetching this enquiry");
  }

  return res.status(200).json({
    success: true,
    data: result.pop()
  })

});

// @desc send bulk email to freely enquiry list
// @api POST /api/v1/admin/enquiry/career/mails
// @access Private ADMIN
const sendBulkEmailsToCareerEnquiry = tryCatch(async (req, res) => {
  const { recipients, templateId, college_id, enquiry_id } = req.body;
  if (!recipients || !templateId || !enquiry_id)
    throw new Error("Some required fields are missing");
  const recipientsFields = [];
  recipients.forEach((recipient) => {
    if (!recipient.email || !recipient.name || !recipient.mobile)
      throw new Error("Some required fields are missing in recipients");
    recipientsFields.push({
      email: recipient.email,
      name: recipient.name,
      mobile: recipient.mobile,
    });
  });

  // Fetch Template details
  const attributesTemp = [
    `${templateTbl}.id`,
    `${templateTbl}.title`,
    `${templateTbl}.subject`,
    `${templateTbl}.contents`,
    `${templateTbl}.type`,
  ];
  const sql = `SELECT ${attributesTemp.join(
    ", "
  )} FROM ${templateTbl} WHERE id = ? LIMIT 1`;
  const result = await dbQuery(sql, [templateId]);
  if (result.length === 0) throw new Error("Invalid template id");

  const { subject, contents, type, id } = result[0];

  if (contents) {
    const college = new RegExp(/\[.*\CAREER-COLLEGE\b.*\]/g);
    if (college.test(contents)) {
      if (!college_id)
        throw new Error("college_id is required for this template");
    }
  }

  let contentWithVariables = contents;
  let sunjectWithVariables = subject;
  let requiredCollegeDetails = null;

  // Fetch College details
  if (college_id) {
    const doesCollegeExist = await checkCollegeExist(college_id);
    if (!doesCollegeExist) throw new Error("Invalid college_id");
    const collegeDetails = await getCollegeBasicDetails(college_id);
    requiredCollegeDetails = collegeDetails;
    // Replace template content variables
    contentWithVariables = contents
      .replace(/\[.*\COLLEGELOGO\b.*\]/g, requiredCollegeDetails.institute_logo)
      .replace(
        /\[.*\CAREER-COLLEGE\b.*\]/g,
        requiredCollegeDetails.institute_name
      );

    sunjectWithVariables = subject.replace(
      /\[.*\CAREER-COLLEGE\b.*\]/g,
      requiredCollegeDetails.institute_name
    );
  }

  // Fetch Career details
  const attributesCareer = [
    `${table}.id`,
    `${courseTbl}.course_name AS course`,
    `${table}.page_url`,
  ];
  const joinTbl = `LEFT JOIN ${courseTbl} ON ${courseTbl}.course_id = ${table}.course`;
  const careerSql = `SELECT ${attributesCareer.join(
    ", "
  )} FROM ${table} ${joinTbl} WHERE id = ? LIMIT 1`;
  const careerResult = await dbQuery(careerSql, [enquiry_id]);
  // if (careerResult.length === 0) throw new Error("Invalid enquiry id");
  const { page_url, course } = careerResult[0];
  // Replace template content variables
  contentWithVariables = contentWithVariables
    .replace(/\[.*\CAREER-URL\b.*\]/g, page_url)
    .replace(/\[.*\CAREER-NAME\b.*\]/g, course);

  sunjectWithVariables = sunjectWithVariables.replace(
    /\[.*\CAREER-NAME\b.*\]/g,
    course
  );

  // CHECK FOR SMS TEMPLATE
  if (type === "sms") {
    const careerSmsTemplate = smsTemplate[id];
    if (!careerSmsTemplate) throw new Error("Invalid template id for sms");
    contentWithVariables = careerSmsTemplate
      .replace(/\[CAREER-URL\]/g, page_url)
      .replace(/\[CAREER-COURSE\]/g, course)
      .replace(/\[CAREER-COLLEGE\]/g, requiredCollegeDetails.institute_name)
      .replace(/\[SCHOOL-URL\]/g, requiredCollegeDetails.institute_url)
      .replace(/\[COLLEGE-NAME\]/g, requiredCollegeDetails.institute_url);

    recipientsFields.forEach(async (recipient) => {
      let smsURL = getSmsDetails(contentWithVariables, recipient.mobile);
      // Changing Names in variables
      smsURL = smsURL.replace(/\[CAREER-NAME\]/g, recipient.name);
      axios({
        method: "GET",
        url: smsURL,
        headers: {
          "Content-Type": "application/x-www-form-urlencoded",
        },
      })
        .then((response) => {
          if (response.data.ErrorCode !== "000")
            throw response.data.ErrorMessage;
        })
        .catch((error) => {
          console.log(error);
          throw new Error(error);
        });
    });

    return res
      .status(200)
      .json({ success: true, message: "SMS/Email sent successfully" });
  }

  // Send mail to all recipients
  recipientsFields.forEach((recipient) => {
    const mailOptions = {
      from: EMAIL_FROM,
      to: recipient.email,
      subject: sunjectWithVariables,
      html: contentWithVariables,
    };
    transporter.sendMail(mailOptions, (error, info) => {
      if (error) {
        return res.status(500).json({
          success: false,
          message: "Error in sending mail",
          error: error.message,
        });
      }
    });
  });

  return res
    .status(200)
    .json({ success: true, message: "Mails sent successfully" });
});

// @desc get all freely enquiry list
// @api GET /api/v1/admin/enquiry/career
// @access Private
const get_freelyEnquiry = tryCatch(async (req, res) => {
 
  const { page = 1, limit = 50, search } = req.query;
  if (Number(page) < 1)
    throw new Error("Page number is required and should be greater than 0");
  const offset = (page - 1) * limit;

  const queryMap = {
    attributes: [
      `${table}.id`,
      `${table}.fullname`,
      `${table}.email`,
      `${table}.contact`,
      `${CAREER_LIST}.career_name AS course`,
      `${table}.page_url`,
      // `${table}.city`,
      // `${table}.state`,

      `${STATE_TBL}.state`,
      `${CITY_TBL}.cname as city`,

      `${table}.date`,
      `${table}.remark`,
      `${table}.remark_by`,
      `${table}.remark_dt`,
      `${table}.active`,
      `${table}.npf_response`,
    ],
    additional: [`ORDER BY ${TBL_FREELY_ENQUIRY}.date DESC`, `LIMIT ${Number(limit) || 10} OFFSET ${offset}`],
    joinTbl: [
      `LEFT JOIN ${CAREER_LIST} ON ${CAREER_LIST}.content_id = ${TBL_FREELY_ENQUIRY}.course`,
      `LEFT JOIN ${STATE_TBL} ON ${STATE_TBL}.sid = ${table}.state`,
      `LEFT JOIN ${CITY_TBL} ON ${CITY_TBL}.ctid = ${table}.city`
      ],
    //  joinTbl: [`LEFT JOIN ${STATE_TBL} ON ${STATE_TBL}.sid = ${table}.state`],
    // joinTbl: [`LEFT JOIN ${CITY_TBL} ON ${STATE_TBL}.sid = ${TBL_FREELY_ENQUIRY}.state`],


    
    conditions: [],
    parameters: [],
    table: TBL_FREELY_ENQUIRY,
  };

  queryMap.attributes.push(
    `CASE WHEN ${TBL_FREELY_ENQUIRY}.city>0  THEN (select cname from ${CITY_TBL} where ctid=${table}.city) ELSE ${TBL_FREELY_ENQUIRY}.city END AS city`
  );

   queryMap.attributes.push(
    `CASE WHEN ${TBL_FREELY_ENQUIRY}.state>0  THEN (select  state from ${STATE_TBL} where sid=${table}.state) ELSE ${TBL_FREELY_ENQUIRY}.state END AS state `
  );


  if(search){
    queryMap.conditions.push(`fullname LIKE ? OR email LIKE ? OR contact LIKE ? OR ${TBL_FREELY_ENQUIRY}.city LIKE ? OR ${TBL_FREELY_ENQUIRY}.state LIKE ?`);
    queryMap.parameters.push(`${search}%`,`${search}%`,`${search}%`,`${search}%`,`${search}%`);
  }



 


  const { result, total } = await getEnquiryPaginationService(queryMap);

  return res.status(200).json({
    success: true,
    data: result,
    records: total,
  })

});

// @desc update freely enquiry list
// @api PUT /api/v1/admin/enquiry/career/:id
// @access Private ADMIN
const update_freelyEnquiry = tryCatch(async (req, res) => {

  const { id } = req.params;

  const { remark, remark_by } = req.body;
  if(!remark, !remark_by){
    res.statusCode = 400;
    throw new Error("Remark or Remark by is missing");
  };

  const queryMap = {
    update: [`remark = ?`, `remark_by = ?`,  `remark_dt = NOW()`].join(", "),
    conditions: [`${TBL_FREELY_ENQUIRY}.id = ?`],
    parameters: [remark, remark_by, id],
    table: TBL_FREELY_ENQUIRY,
  }

  const result = await updateEnquiryService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while updating the enquiry")
  }

  return res.status(200).json({
    success: true,
    message: "Enquiry updated successfully"
  })

});

// @desc delete freely enquiry list
// @api DELETE /api/v1/admin/enquiry/career/:id
// @access Private ADMIN
const del_freelyEnquiry = tryCatch(async (req, res) => {

  const { id } = req.params;

  const queryMap = {
    conditions: [`${TBL_FREELY_ENQUIRY}.id = ?`],
    parameters: [id],
    table: TBL_FREELY_ENQUIRY
  }

  const result = await deleteEnquiryService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while deleting the Enquiry")
  };

  return res.status(200).json({
    success: true,
    message: "Enquiry deleted successfully"
  })

});

module.exports = {
  get_freelyEnquiry,
  update_freelyEnquiry,
  del_freelyEnquiry,
  getFreelyEnquiryDetails,
  sendBulkEmailsToCareerEnquiry,
};
