const tryCatch = require("../../../../utils/tryCatch");
const updateColumn = require("../../../../utils/updateColumn");
const { SC_SCHOOLLOCATIONMASTER } = require("../../../../models/tables");
const { getSchoolDetailsPaginationService, addNewSchoolDetailsService, deleteSchoolDetailsService, updateSchoolDetailsService } = require("../../../../services/schoolDetails.service");
const { areAllKeysPresent } = require("../../../../utils/common_utils");

// @desc get all School Locations
// @api GET /api/admin/school-location
// @access Private
const get_schoolLocation = tryCatch(async (req, res) => {

  const { page = 1, limit = 50, search } = req.query;
  if(Number(page) < 1) throw new Error("Page number is required and should be greater than 0");
  const offset = (page - 1) * limit;

  const queryMap = {
    attributes: [`${SC_SCHOOLLOCATIONMASTER}.*`],
    additional: [`ORDER BY ${SC_SCHOOLLOCATIONMASTER}.dateModify DESC`, `LIMIT ${Number(limit) || 10} OFFSET ${offset}`],
    conditions: [],
    parameters: [],
    table: SC_SCHOOLLOCATIONMASTER,
  };

  if(search){
    queryMap.conditions.push(`cityName LIKE ?`);
    queryMap.parameters.push(`${search}%`);
  };

  const { result, total } = await getSchoolDetailsPaginationService(queryMap);

  return res.status(200).json({
    success: true,
    data: result,
    records: total,
  });

});

// @desc post School Location
// @api POST /api/admin/school-location
// @access Private ADMIN
const post_schoolLocation = tryCatch(async (req, res) => {

  const requiredFields = setSchoolLocationAttributes().required;
  if(areAllKeysPresent(req.body, requiredFields)){
    res.statusCode = 400;
    throw new Error(`Required fields are ${requiredFields.join(", ")}`);
  };

  const queryMap = {
    attributes: setSchoolLocationAttributes().attributes.concat(["dateAdded", "addedBy", "priority"]),
    parameters: setSchoolLocationAttributes().attributes.map((field) => req.body[field]).concat([(new Date()).getTime(), req.adminId, (req.body.priority || 0)]),
    table: SC_SCHOOLLOCATIONMASTER,
  };

  const result = await addNewSchoolDetailsService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Failed to add school location");
  };

  return res.status(200).json({
    success: true,
    message: "school location added",
  });

});

// @desc update School Location
// @api PUT /api/admin/school-location
// @access Private ADMIN
const update_schoolLocation = tryCatch(async (req, res) => {

  const { cityId, update } = req.body;
  if(!Number(cityId) < 0 || !update){
    res.statusCode = 400;
    throw new Error("Invalid data, cityId or update required");
  };

  const { column, value } = updateColumn(update);

  const queryMap = {
    update: [column, `dateModify = NOW(), modifyBy = ?`],
    conditions: [`cityId = ?`],
    parameters: [...value, req.adminId, cityId],
    table: SC_SCHOOLLOCATIONMASTER,
  };

  const result = await updateSchoolDetailsService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Failed to update city");
  };

  return res.status(200).json({
    success: true,
    message: "school location updated",
  });

});

// @desc delete School Location
// @api DELETE /api/admin/school-location
// @access Private ADMIN
const del_schoolLocation = tryCatch(async (req, res) => {

  const { cityId } = req.body;
  if(!Number(cityId) < 0){
    res.statusCode = 400;
    throw new Error("Invalid data, cityId required");
  }

  const queryMap = {
    conditions: [`cityId = ?`],
    parameters: [cityId],
    table: SC_SCHOOLLOCATIONMASTER,
  };

  const result = await deleteSchoolDetailsService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Failed to delete city");
  }

  return res.status(200).json({
    success: true,
    message: "school location deleted",
  });

});

module.exports = {
  get_schoolLocation,
  post_schoolLocation,
  update_schoolLocation,
  del_schoolLocation,
};


const setSchoolLocationAttributes = () => {
  return (
    {
      required: ["cityName", "parentId"],
      attributes: [
        "cityName",
        "parentId",
        "cityMatchName",
        "metaTitle",
        "metaDescription",
        "metaKeywords",
        "priority",
      ],
    }
  )
};