const tryCatch = require("../../../../utils/tryCatch");
const updateColumn = require("../../../../utils/updateColumn");
const { SC_SCHOOLCITYMASTER } = require("../../../../models/tables");
const { getSchoolDetailsPaginationService, deleteSchoolDetailsService, updateSchoolDetailsService, addNewSchoolDetailsService } = require("../../../../services/schoolDetails.service");
const { areAllKeysPresent } = require("../../../../utils/common_utils")

// @desc  Get all school state city
// @route GET /api/v1/admin/school-city
// @access  Private
const get_schoolCity = tryCatch(async (req, res) => {
  
  const { page = 1, limit = 50, search } = req.query;
  if(Number(page) < 1) throw new Error("Page number is required and should be greater than 0");
  const offset = (page - 1) * limit;

  const queryMap = {
    attributes: [`${SC_SCHOOLCITYMASTER}.*`],
    additional: [`ORDER BY ${SC_SCHOOLCITYMASTER}.dateModify DESC`, `LIMIT ${limit} OFFSET ${offset}`],
    conditions: [`cityLevel = 1`],
    parameters: [],
    table: SC_SCHOOLCITYMASTER,
  };

  if(search){
    queryMap.conditions.push(`cityName LIKE ?`);
    queryMap.parameters.push(`${search}%`);
  };

  const { result, total } = await getSchoolDetailsPaginationService(queryMap);

  return res.status(200).json({
    success: true,
    data: result,
    records: total,
  });

});

// @desc Post new school city
// @route POST /api/v1/admin/school-city
// @access  Private ADMIN
const post_schoolCity = tryCatch(async (req, res) => {

  const requiredFields = setStateCityAttributes().required;
  if(!areAllKeysPresent(req.body, requiredFields)){
    res.statusCode = 400;
    throw new Error("Invalid data, required fields are missing");
  }

  const queryMap = {
    attributes: setStateCityAttributes().attributes.concat(["dateAdded", "addedBy", "priority"]),
    parameters: setStateCityAttributes().attributes.map((field) => req.body[field]).concat([new Date(), req.adminId, (req.body.priority || "")]),
    table: SC_SCHOOLCITYMASTER,
  };

  const result = await addNewSchoolDetailsService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Failed to add new city");
  };

  return res.status(200).json({
    success: true,
    message: "New city added",
  });

});

// @desc Update school city
// @route PUT /api/v1/admin/school-city
// @access  Private ADMIN
const update_schoolCity = tryCatch(async (req, res) => {

  const { cityId, update } = req.body;
  if(!Number(cityId) < 0 || !update){
    res.statusCode = 400;
    throw new Error("Invalid data, cityId or update required");
  };

  const { column, value } = updateColumn(update);

  const queryMap = {
    update: [column, `dateModify = NOW(), modifyBy = ?`],
    conditions: [`cityId = ?`],
    parameters: [...value, req.adminId, cityId],
    table: SC_SCHOOLCITYMASTER,
  };

  const result = await updateSchoolDetailsService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Failed to update city");
  };

  return res.status(200).json({
    success: true,
    message: "City updated",
  });

});

// @desc Delete school city
// @route DELETE /api/v1/admin/school-city
// @access  Private ADMIN
const del_schoolCity = tryCatch(async (req, res) => {

  const { cityId } = req.body;
  if(!Number(cityId) < 0){
    res.statusCode = 400;
    throw new Error("Invalid data, cityId required");
  }

  const queryMap = {
    conditions: [`cityId = ?`],
    parameters: [cityId],
    table: SC_SCHOOLCITYMASTER,
  };

  const result = await deleteSchoolDetailsService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Failed to delete city");
  }

  return res.status(200).json({
    success: true,
    message: "City deleted",
  });

});

module.exports = {
  get_schoolCity,
  post_schoolCity,
  update_schoolCity,
  del_schoolCity,
};


const setStateCityAttributes = () => {
  return (
    {
      required: ["cityName", "parentId", "top_city"],
      attributes: [
        "cityName",
        "cityMatchName",
        "parentId",
        "top_city",
        "metaTitle",
        "metaDescription",
        "metaKeywords",
        "cityLevel",
      ],
    }
  )
};