const tryCatch = require("../../../../utils/tryCatch");
const updateColumn = require("../../../../utils/updateColumn");
const { TBL_AWARDS } = require("../../../../models/tables");
const { getSchoolDetailsService, addNewSchoolDetailsService, updateSchoolDetailsService, deleteSchoolDetailsService } = require("../../../../services/schoolDetails.service");
const { areAllKeysPresent } = require("../../../../utils/common_utils");

// @desc get school award details by school id
// @route GET /api/v1/admin/awards/:sid?type=(school|college)
// @access Private ADMIN
const getSchoolAwards = tryCatch(async (req, res) => {

  const { sid } = req.params;

  const { type } = req.query;
  if(!type){
    res.statusCode = 400;
    throw new Error("Type is required, or type is invalid");
  };

  const queryMap = {
    attributes: [
      `${TBL_AWARDS}.id AS awd_id`,
      `${TBL_AWARDS}.awd_title`,
      `${TBL_AWARDS}.awd_descriptipn`,
      `${TBL_AWARDS}.is_active`,
    ],
    conditions: [`${TBL_AWARDS}.e_id = ?`, `${TBL_AWARDS}.type = ?`],
    parameters: [sid, type],
    additional: [`ORDER BY ${TBL_AWARDS}.date DESC`],
    table: TBL_AWARDS,
  };

  const result = await getSchoolDetailsService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Failed to get awards");
  };

  return res.status(200).json({
    success: true,
    data: result,
  });

});

// @desc add school award details by school id
// @route POST /api/v1/admin/awards/:sid?type=(school|college)
// @access Private ADMIN
const postSchoolAwards = tryCatch(async (req, res) => {

  const { sid } = req.params;

  const { type } = req.query;
  if(!type){
    res.statusCode = 400;
    throw new Error("Type is required, or type is invalid");
  };

  const { awd_title, awd_descriptipn, is_active } = req.body;

  if(!awd_title){
    res.statusCode = 400;
    throw new Error("awd_title is required");
  }

  const queryMap = {
    attributes: ["awd_title", "awd_descriptipn", "is_active", "type", "e_id", "date"],
    parameters: [awd_title, awd_descriptipn, (is_active || "N"), type, sid, new Date()],
    table: TBL_AWARDS,
  };

  const result = await addNewSchoolDetailsService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Failed to add awards");
  };

  return res.status(200).json({
    success: true,
    message: "Awards added Successfully",
  });

});

// @desc update school award details by school id
// @route PUT /api/v1/admin/awards/:sid?type=(school|college)
// @access Private ADMIN
const updateSchoolAwards = tryCatch(async (req, res) => {

  const { sid } = req.params;

  const { type } = req.query;
  if(!type){
    res.statusCode = 400;
    throw new Error("Type is required, or type is invalid");
  };

  const { update, awd_id } = req.body;
  if(!Number(awd_id) < 0 || !update){
    res.statusCode = 400;
    throw new Error("Invalid data, awd_id or update required");
  };

  delete update.awd_id; // we are getting this from req.body

  const { column, value } = updateColumn(update);

  const queryMap = {
    update: [column, `date = NOW()`].join(", "),
    conditions: [`${TBL_AWARDS}.e_id = ?`, `${TBL_AWARDS}.id = ?`, `${TBL_AWARDS}.type = ?`],
    parameters: [...value, sid, awd_id, type],
    table: TBL_AWARDS,
  };

  const result = await updateSchoolDetailsService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Failed to update awards");
  };

  return res.status(200).json({
    success: true,
    message: "Awards updated Successfully",
  });

});

// @desc delete school award details by school id
// @route DELETE /api/v1/admin/awards/:sid?type=(school|college)
// @access Private ADMIN
const deleteSchoolAwards = tryCatch(async (req, res) => {
 
  const { sid } = req.params;

  const { type } = req.query;
  if(!type){
    res.statusCode = 400;
    throw new Error("Type is required, or type is invalid");
  };

  const { awd_id } = req.body;
  if(!Number(awd_id) < 0){
    res.statusCode = 400;
    throw new Error("Invalid data, awd_id required");
  };

  const queryMap = {
    conditions: [`${TBL_AWARDS}.id = ?`, `${TBL_AWARDS}.e_id = ?`, `${TBL_AWARDS}.type = ?`],
    parameters: [awd_id, sid, type],
    table: TBL_AWARDS,
  };

  const result = await deleteSchoolDetailsService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Failed to delete awards");
  };

  return res.status(200).json({
    success: true,
    message: "Awards deleted Successfully",
  });

});

module.exports = {
  getSchoolAwards,
  postSchoolAwards,
  updateSchoolAwards,
  deleteSchoolAwards,
}
