const tryCatch = require("../../../../utils/tryCatch");
const { TBL_ACCORDION_CONTENT } = require("../../../../models/tables");
const { getSchoolDetailsService, addNewSchoolDetailsService, deleteSchoolDetailsService, updateSchoolDetailsService } = require("../../../../services/schoolDetails.service");
const updateColumn = require("../../../../utils/updateColumn");

//@desc get school FAQ details by school id
//@route GET /api/v1/admin/faq/:sid?type=(school|college)
//@access Private ADMIN
const getSchoolFAQ = tryCatch(async (req, res) => {

  const { sid } = req.params;

  const { type } = req.query;
  if(!type){
    res.statusCode = 400;
    throw new Error("Type is required, or type is invalid");
  };

  const queryMap = {
    attributes: [
      `${TBL_ACCORDION_CONTENT}.id AS faq_id`,
      `${TBL_ACCORDION_CONTENT}.question`,
      `${TBL_ACCORDION_CONTENT}.answer`,
      `${TBL_ACCORDION_CONTENT}.active`,
      `${TBL_ACCORDION_CONTENT}.date`,
    ],
    conditions: [`${TBL_ACCORDION_CONTENT}.org_id = ?`, `${TBL_ACCORDION_CONTENT}.org_type = ?`],
    parameters: [sid, type],
    additional: [`ORDER BY ${TBL_ACCORDION_CONTENT}.date DESC`],
    table: TBL_ACCORDION_CONTENT,
  };

  const result = await getSchoolDetailsService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Failed to get awards");
  };

  return res.status(200).json({
    success: true,
    data: result,
  });

});

// @desc add school FAQ details by school id
// @route POST /api/v1/admin/faq/:sid?type=(school|college)
// @access Private ADMIN
const addSchoolFAQ = tryCatch(async (req, res) => {

  const { sid } = req.params;

  const { type } = req.query;
  if(!type){
    res.statusCode = 400;
    throw new Error("Type is required, or type is invalid");
  };

  const { question, answer, active } = req.body;
  if(!question || !answer){
    res.statusCode = 400;
    throw new Error("Question and Answer are required");
  };

  const queryMap = {
    attributes: ["org_id", "org_type", "question", "answer", "date", "active"],
    parameters: [sid, type, question, answer, new Date(), (active ? "Y" : "N")],
    table: TBL_ACCORDION_CONTENT,
  };

  const result = await addNewSchoolDetailsService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Failed to add awards");
  };

  return res.status(200).json({
    success: true,
    message: "Awards added Successfully",
  });

});

// @desc update school FAQ details by school id
// @route PUT /api/v1/admin/faq/:sid
// @access Private ADMIN
const updateSchoolFAQ = tryCatch(async (req, res) => {
 
  const { sid } = req.params;

  const { type } = req.query;
  if(!type){
    res.statusCode = 400;
    throw new Error("Type is required, or type is invalid");
  };

  const { update, faq_id } = req.body;
  if(!Number(faq_id) < 0 || !update){
    res.statusCode = 400;
    throw new Error("Invalid data, faq_id or update required");
  };

  delete update.faq_id; // we are getting this from req.body

  const  { column, value } = updateColumn(update);

  const queryMap = {
    update: [column, `date = NOW()`].join(", "),
    conditions: [`${TBL_ACCORDION_CONTENT}.org_id = ?`, `${TBL_ACCORDION_CONTENT}.id = ?`, `${TBL_ACCORDION_CONTENT}.org_type = ?`],
    parameters: [...value, sid, faq_id, type],
    table: TBL_ACCORDION_CONTENT,
  }

  const result = await updateSchoolDetailsService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Failed to update awards");
  };

  return res.status(200).json({
    success: true,
    message: "Awards updated Successfully",
  });

});

// @desc delete school FAQ details by school id
// @route DELETE /api/v1/admin/faq/:sid?type=(school|college)
// @access Private ADMIN
const deleteSchoolFAQ = tryCatch(async (req, res) => {

  const { sid } = req.params;

  const { type } = req.query;
  if(!type){
    res.statusCode = 400;
    throw new Error("Type is required, or type is invalid");
  };

  const { faq_id } = req.body;
  if(!Number(faq_id) < 0){
    res.statusCode = 400;
    throw new Error("Invalid faq id")
  }

  const queryMap = {
    conditions: [`${TBL_ACCORDION_CONTENT}.org_id = ?`, `${TBL_ACCORDION_CONTENT}.id = ?`, `${TBL_ACCORDION_CONTENT}.org_type = ?`],
    parameters: [sid, faq_id, type],
    table: TBL_ACCORDION_CONTENT,
  }

  const result = deleteSchoolDetailsService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Failed to delete awards");
  };

  return res.status(200).json({
    success: true,
    message: "Awards deleted Successfully",
  });

});

module.exports = {
  getSchoolFAQ,
  addSchoolFAQ,
  updateSchoolFAQ,
  deleteSchoolFAQ,
};