const { dbQuery } = require("../../../../config/db.config");
const tryCatch = require("../../../../utils/tryCatch");
const updateColumn = require("../../../../utils/updateColumn");
const { errorMsgEmpty } = require("../../../../utils/errors_message");
const { TBL_SCHOOL_GRADE_FEE } = require("../../../../models/tables");
const { checkSchoolGradeFeesExist, checkSchoolExist } = require("../../../../utils/leadAccessCheck");
const { splitByUnderscoresAndCapitalize } = require("../../../../utils/common_utils");

const table = TBL_SCHOOL_GRADE_FEE;

// @desc get school grade details by school id
// @route GET /api/v1/admin/grade-fees/:sid
// @access Private ADMIN
const getSchoolGradeFees = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { sid } = req.params;
  if(!sid) throw new Error("School id is required");
  const doesSchoolExist = await checkSchoolExist(sid);
  if(!doesSchoolExist) throw new Error("School doesn't exist")
  const doesSchoolGradesFeesExists = await checkSchoolGradeFeesExist(sid);
  if(!doesSchoolGradesFeesExists) errorMsgEmpty("School grade fees not found",res);
  const attributes = [
    `${table}.id`,
    `${table}.grade`,
    `${table}.fee`,
  ]
  sql = `SELECT ${attributes.join(", ")}  FROM ${table} WHERE sid = ? ORDER BY ${table}.sort ASC`;
  result = await dbQuery(sql, [sid]);

  return res.status(200).json({
    success: true,
    data: result,
  });
});

// @desc add school grade details by school id
// @route POST /api/v1/admin/grade-fees/:sid
// @access Private ADMIN
const addSchoolGradeFees = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];

  const { sid } = req.params;
  if(!sid) throw new Error("School id is required");
  const { grade_fee } = req.body;
  if(!grade_fee) throw new Error("Grade and fee is required");

  const doesSchoolGradesFeesExists = await checkSchoolGradeFeesExist(sid);
  if(doesSchoolGradesFeesExists) {
   const deleteExistingGradesFeesSql = `DELETE FROM ${table} WHERE sid = ?`;
   const deleteExistingGradesFeesResult = await dbQuery(deleteExistingGradesFeesSql, [sid]);
   if(!deleteExistingGradesFeesResult.affectedRows) throw new Error("Something went wrong, while deleting existing school grade fees");
  }
  
  const gradesFeesSortObj = getGradesFeesSortObj(grade_fee);
  const attributes = [
    `${table}.sid`,
    `${table}.grade`,
    `${table}.fee`,
    `${table}.sort`,
  ];
  const values = Object.values(gradesFeesSortObj).map((obj) => {
    return [sid, obj.grade, obj.fee, obj.sort];
  });
  sql = `INSERT INTO ${table} (${attributes.join(", ")}) VALUES ?`;
  result = await dbQuery(sql, [values]);

  if (!result.affectedRows) throw new Error("Something went wrong while adding school grade fees");

  return res.status(200).json({
    success: true,
    message: "School grade fees added successfully",
  });

});

const getGradesFeesSortObj = (grade_fee) => {
  if (Object.keys(grade_fee).length === 0)
    throw new Error("Grade and fee is required");
  const gradesFeesSortObj = {};
  const requiredGradesSort = {
    "nursery": 1,
    "lkg": 2,
    "ukg": 3,
    "grade_1": 4,
    "grade_2": 5,
    "grade_3": 6,
    "grade_4": 7,
    "grade_5": 8,
    "grade_6": 9,
    "grade_7": 10,
    "grade_8": 11,
    "grade_9": 12,
    "grade_10": 13,
    "grade_11": 14,
    "grade_12": 15,
  };

 Object.keys(grade_fee).forEach((grade) => {
  if(requiredGradesSort[grade]){
    gradesFeesSortObj[grade] = {
      grade: splitByUnderscoresAndCapitalize(grade),
      fee: grade_fee[grade],
      sort: requiredGradesSort[grade],
    };
  }
 });

  return gradesFeesSortObj;
}

module.exports = {
  getSchoolGradeFees,
  addSchoolGradeFees,
  getGradesFeesSortObj
}