const tryCatch = require("../../../../utils/tryCatch");
const updateColumn = require("../../../../utils/updateColumn");
const { urlExist, postUrlExist } = require("../../../../utils/url_exist");
const { SCHOOL_DETAIL, SC_SCHOOLCITYMASTER } = require("../../../../models/tables");
const { schoolCreateFields } = require("./create_school_content");
const {
  getSchoolDetailsService,
  getSchoolDetailsPaginationService,
  addNewSchoolService,
  deleteSchoolService,
  updateSchoolService,
} = require("../../../../services/school.service");
const { areAllKeysPresent } = require("../../../../utils/common_utils");

// @desc get school details by id
// @api GET /api/admin/school-detail/:sid
// @access Private ADMIN
const getSchoolDetailsById = tryCatch(async (req, res) => {

  const { sid } = req.params;

  const queryMap = {
    attributes: [`${SCHOOL_DETAIL}.*`],
    conditions: [`${SCHOOL_DETAIL}.sid = ?`],
    parameters: [sid],
    additional:[`LIMIT 1`]
  };

  const result = await getSchoolDetailsService(queryMap);
  if(result instanceof Error || !result[0]){
    res.statusCode = 400;
    throw new Error("No data found for this school")
  }

  return res.status(200).json({
    success: true,
    data: result.pop(),
  })

});

// @desc update school details
// @api PUT /api/admin/school-detail/:sid
// @access Private ADMIN
const updateSchoolDetailsById = tryCatch(async (req, res) => {
  


const othercity=req.body.update.other_city?req.body.update.other_city:''
 

  const { sid } = req.params;
  const { update } = req.body;
  if(!update) throw new Error("Please provide sid and update data");

  update.school_url && (await urlExist(SCHOOL_DETAIL, "sid", "school_url", update.school_url, sid));

  const requiredFields = {};
  [...schoolCreateFields, "priority", "active"].forEach((field) => {
    if (update[field]) requiredFields[field] = update[field];
  });

  
  const { column, value } = updateColumn(requiredFields);
  
  const queryMap = {
    update: [column, "update_date = NOW()",`other_city='${othercity}'`].join(", "),
    conditions: [`${SCHOOL_DETAIL}.sid = ?`],
    parameters: [...value, sid]
  }
  const result = await updateSchoolService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while updating the school")
  }

  return res.status(200).json({
    success: true,
    message: "Successfully updated the school"
  })

});

// @desc delete school details
// @api DELETE /api/admin/school-detail/:sid
// @access Private ADMIN
const deleteSchoolDetailsById = tryCatch(async (req, res) => {

  const { sid } = req.params;

  const queryMap = {
    conditions: [`${SCHOOL_DETAIL}.sid = ?`],
    parameters: [sid]
  };

  const result = await deleteSchoolService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while deleting school")
  }

  return res.status(200).json({
    success:true,
    message: "School deleted successfully"
  })

});

// @desc get all school details
// @api GET /api/admin/school-details
// @access Private ADMIN
const get_schoolDetails = tryCatch(async (req, res) => {
  const { page = 1, limit = 50, search, partner } = req.query;

  if (Number(page) < 1)
    throw new Error("Page number is required and should be greater than 0");
  const offset = Number((page - 1) * limit);

  const queryMap = {
    attributes: getSchoolDetailsAttributes().attributes,
    additional: [
      `ORDER BY ${SCHOOL_DETAIL}.add_date DESC`,
      `LIMIT ${Number(limit) || 10} OFFSET ${offset}`,
    ],
    joinTbl: [`LEFT JOIN ${SC_SCHOOLCITYMASTER} ON (${SC_SCHOOLCITYMASTER}.cityId = ${SCHOOL_DETAIL}.state OR ${SC_SCHOOLCITYMASTER}.cityName = ${SCHOOL_DETAIL}.state)`],
    conditions: [],
    parameters: [],
  };

  if (search) {
    queryMap.conditions.push(`${SCHOOL_DETAIL}.school_name LIKE ? OR ${SCHOOL_DETAIL}.school_url  LIKE ?`);
    queryMap.parameters.push(`%${search}%`,`%${search}%`);
  }

  if (partner) {
    queryMap.conditions.push(`${SCHOOL_DETAIL}.admission_partner LIKE ?`);
    queryMap.parameters.push(`%${partner}%`);
  }

  const { result, total } = await getSchoolDetailsPaginationService(queryMap);

  return res.status(200).json({
    success: true,
    data: result,
    records: total,
  });
});

// @desc post new school details
// @api POST /api/admin/school-details
// @access Private ADMIN
const post_schoolDetails = tryCatch(async (req, res) => {

  const requiredFields = ["school_name", "school_url", "show_on_home"];
  if(!areAllKeysPresent(req.body, requiredFields)){
    res.statusCode = 400;
    throw new Error("Some fields are missing")
  };

  req.body.school_url && (await postUrlExist(SCHOOL_DETAIL, "school_url", req.body.school_url));

  const queryMap = {
    attributes: [...schoolCreateFields, "priority", "active", "added_by", "updated_by", "add_date", "update_date"],
    parameters: [...schoolCreateFields.map((field) => req.body[field]), (req.body.priority || 20), (req.body.active || "Y"), "admin", "admin", new Date(), new Date()]
  }

  const result = await addNewSchoolService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while adding school");
  }

  return res.status(200).json({
    success: true,
    message: "New school added successfully",
    sid: result.insertId
  })

});

module.exports = {
  get_schoolDetails,
  post_schoolDetails,
  getSchoolDetailsById,
  updateSchoolDetailsById,
  deleteSchoolDetailsById,
};

const getSchoolDetailsAttributes = () => {
  return {
    attributes: [
      `${SCHOOL_DETAIL}.sid`,
      `${SCHOOL_DETAIL}.school_name`,
      `${SCHOOL_DETAIL}.school_url`,
      `${SCHOOL_DETAIL}.show_on_home`,
      `${SCHOOL_DETAIL}.contact_phone`,
      `${SCHOOL_DETAIL}.contact_mobile`,
      `${SC_SCHOOLCITYMASTER}.cityName AS state`,
      `${SCHOOL_DETAIL}.city`,
      `${SCHOOL_DETAIL}.add_date`,
      `${SCHOOL_DETAIL}.update_date`,
      `${SCHOOL_DETAIL}.school_logo`,
    ],
  };
};
