const tryCatch = require("../../../utils/tryCatch");
const updateColumn = require("../../../utils/updateColumn");
const { urlExist, postUrlExist } = require("../../../utils/url_exist");
const {
  CAREER_LIST,
  CATEGORY,
  COURSE,
  SPECIALIZATION,
} = require("../../../models/tables");
const { getCareerPaginationService, deleteCareerService, updateCareerService, addCareerService } = require("../../../services/career.service");
const { areAllKeysPresent } = require("../../../utils/common_utils")

// @desc get all career list
// @api GET /api/v1/admin/career
// @access Private
const get_careerList = tryCatch(async (req, res) => {
 
  const { page = 1, limit = 50, search } = req.query;
  if(Number(page) < 1) throw new Error("Page number is required and should be greater than 0");
  const offset = (page - 1) * limit;

  const queryMap = {
    attributes: [
      `${CAREER_LIST}.*`,
      `${COURSE}.course_name`,
      `${CATEGORY}.cat_name`,
      `${SPECIALIZATION}.spce_name`,
    ],
    joinTbl: [
      `LEFT JOIN ${CATEGORY} ON ${CATEGORY}.cat_id = ${CAREER_LIST}.category`,
      `LEFT JOIN ${COURSE} ON ${COURSE}.course_id = ${CAREER_LIST}.courseid`,
      `LEFT JOIN ${SPECIALIZATION} ON ${SPECIALIZATION}.spec_id = ${CAREER_LIST}.specialization`,
    ],
    additional: [`ORDER BY ${CAREER_LIST}.update_date DESC`, `LIMIT ${Number(limit) || 10} OFFSET ${offset}`],
    conditions: [],
    parameters: [],
  };

  if(search){
    queryMap.conditions.push(`${CAREER_LIST}.career_name LIKE ?`);
    queryMap.parameters.push(`${search}%`);
  }

  if(req.query.category){
    queryMap.conditions.push(`${CATEGORY}.cat_name LIKE ?`);
    queryMap.parameters.push(`${req.query.category}%`);
  }

  if(req.query.course){
    queryMap.conditions.push(`${COURSE}.course_name LIKE ?`);
    queryMap.parameters.push(`${req.query.course}%`);
  }

  if(req.query.specialization){
    queryMap.conditions.push(`${SPECIALIZATION}.spce_name LIKE ?`);
    queryMap.parameters.push(`${req.query.specialization}%`);
  }

  if(req.query.id){
    queryMap.conditions = [`${CAREER_LIST}.content_id = ?`];
    queryMap.parameters = [req.query.id];
    queryMap.additional = [`LIMIT 1`];
  }

  const { result, total } = await getCareerPaginationService(queryMap);

  return res.status(200).json({
    success: true,
    data: result,
    records: total,
  });

});

// @desc post career list
// @api POST /api/v1/admin/career
// @access Private ADMIN
const post_careerList = tryCatch(async (req, res) => {

  

  const requiredFields = setCareerListAttributes().required;
  if(!areAllKeysPresent(req.body, requiredFields)){
    res.statusCode = 400;
    throw new Error("Some mandatory fields are missing");
  }

  await postUrlExist(CAREER_LIST, "career_url", req.body.career_url);

  const queryMap = {
    attributes: setCareerListAttributes().attributes.concat(["publish_date", "update_date", "priority"]),
    parameters: setCareerListAttributes().attributes.map((field) => req.body[field]).concat([new Date(), new Date(), (req.body.priority || "")]),
  };

  const result = await addCareerService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error! cant insert new career");
  };

  return res.status(200).json({ success: true, message: "Career created" });

});

// @desc update career list
// @api api/v1/admin/career
// @access Private ADMIN
const update_careerList = tryCatch(async (req, res) => {

  const { content_id, update } = req.body;
  if(!Number(content_id) < 0 || !update){
    res.statusCode = 400;
    throw new Error("Content ID or Update is required");
  }

  update.career_url && await urlExist(CAREER_LIST, "content_id", "career_url", update.career_url, content_id);

  const { column, value } = updateColumn(update);

  const queryMap = {
    update: [column, `update_date = NOW()`].join(", "),
    conditions: [`${CAREER_LIST}.content_id = ?`],
    parameters: [...value, content_id],
  };

  const result  = await updateCareerService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error! cant update career");
  };

  return res.status(200).json({ success: true, message: "Career updated" });

});

// @desc delete career list
// @api api/v1/admin/career
// @access Private ADMIN
const del_careerList = tryCatch(async (req, res) => {

  const { content_id } = req.body;
  if(!Number(content_id) < 0){
    res.statusCode = 400;
    throw new Error("Content ID is required");
  }

  const queryMap = {
    conditions: [`${CAREER_LIST}.content_id = ?`],
    parameters: [content_id],
  };

  const result = await deleteCareerService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error! cant delete career");
  }

  return res.status(200).json({ success: true, message: "Career deleted" });

});

module.exports = {
  get_careerList,
  post_careerList,
  update_careerList,
  del_careerList,
};


const setCareerListAttributes = () => {
  return (
    {
      required: ["category", "courseid", "specialization", "career_name", "show_on_home", "course_level", "large_content", "career_url"],
      attributes: [
        // "content_id",
        "category",
        "courseid",
        "specialization",
        "career_name",
        "long_title",
        "video_url",
        "career_url",
        "show_on_home",
        "career_logo",
        "career_details_image",
        "content",
        "jobs_aspects",
        "academic_pressure",
        "job_pressure",
        "small_content",
        "early_salary_start",
        "early_salary_end",
        "midlevel_salary_start",
        "midlevel_salary_end",
        "seniorlevel_salary_start",
        "seniorlevel_salary_end",
        "large_content",
        "meta_title",
        "meta_keyword",
        "meta_description",
        "course_level",
        "ex_college_url"
      ]
    }
  )
}