const tryCatch = require("../../../utils/tryCatch");
const updateColumn = require("../../../utils/updateColumn");
const { urlExist, postUrlExist } = require("../../../utils/url_exist");
const { TBL_EXAM_COURSE, CATEGORY, COURSE } = require("../../../models/tables");
const { getExamPaginationService, updateExamService, deleteExamService, addExamService } = require("../../../services/exam.service");


// @desc get all the exam course + other details to use in updating exam course
// @api GET /api/admin/exam-course
// @access Private
const get_examList = tryCatch(async (req, res) => {

  const { page = 1, limit = 50, search } = req.query;
  if(Number(page) < 1) throw new Error("Page number is required and should be greater than 0");
  const offset = (page - 1) * limit;

  const queryMap = {
    attributes: [
      `${TBL_EXAM_COURSE}.*`,
      `${COURSE}.course_name`,
      `${CATEGORY}.cat_name`,
    ],
    joinTbl: [
      `LEFT JOIN ${CATEGORY} ON ${CATEGORY}.cat_id = ${TBL_EXAM_COURSE}.category`,
      `LEFT JOIN ${COURSE} ON ${COURSE}.course_id = ${TBL_EXAM_COURSE}.course_id`,
    ],
    additional: [`ORDER BY ${TBL_EXAM_COURSE}.content_id DESC`, `LIMIT ${Number(limit) || 10} OFFSET ${offset}`],
    conditions: [],
    parameters: [],
  };

  if(search){
    queryMap.conditions.push(`${TBL_EXAM_COURSE}.exam_name LIKE ?`);
    queryMap.parameters.push(`${search}%`);
  };

  if(req.query.category){
    queryMap.conditions.push(`${CATEGORY}.cat_name LIKE ?`);
    queryMap.parameters.push(`${req.query.category}%`);
  }

  if(req.query.id){
    queryMap.conditions = [`${TBL_EXAM_COURSE}.content_id = ?`];
    queryMap.parameters = [req.query.id];
    queryMap.additional = [`LIMIT 1`];
  };

  const { result, total } = await getExamPaginationService(queryMap);

  return res.status(200).json({ success: true, data: result, records: total });

});

// @desc post new exam course
// @api POST /api/admin/exam-course
// @access Private ADMIN
const post_examList = tryCatch(async (req, res) => {


   
  
 
  const requiredFields = setExamAttributes().required;
  // if(!areAllKeysPresent(requiredFields, req.body)){
  //   res.statusCode = 400;
  //   throw new Error("Some required fields are missing");
  // }

  await postUrlExist(TBL_EXAM_COURSE, "exam_url", req.body.exam_url);  
  const queryMap = {
    attributes: setExamAttributes().attributes.concat(["priority",  "course_id"]),
    parameters: setExamAttributes().attributes.map((field) => req.body[field]).concat([(req.body.priority || 20), (req.body.course_id || 0)]),
  };

  const result = await addExamService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while adding new exam");
  };

  return res.status(200).json({ success: true, message: "Exam list added successfully" });

});

// @desc update exam list
// @api PUT /api/admin/exam-list
// @access Private ADMIN
const update_examList = tryCatch(async (req, res) => {

  const { content_id, update } = req.body;
  if(!Number(content_id) < 0 || !update){
    res.statusCode = 400;
    throw new Error("content_id or update is invalid or missing");
  };

  update.exam_url && await urlExist(TBL_EXAM_COURSE, "content_id", "exam_url", update.exam_url, content_id);

  const { column, value } = updateColumn(update);

  const queryMap = {
    update: column,
    conditions: [`${TBL_EXAM_COURSE}.content_id = ?`],
    parameters: [...value, content_id],
  };

  const result = await updateExamService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while updating exam");
  };

  return res.status(200).json({ success: true, message: "Exam list updated successfully" });

});

// @desc delete exam list
// @api DELETE /api/admin/exam-list
// @access Private ADMIN
const del_examList = tryCatch(async (req, res) => {

  const { content_id } = req.body;
  if(!Number(content_id) < 0){
    res.statusCode = 400;
    throw new Error("content_id is invalid or missing");
  }

  const queryMap = {
    conditions: [`${TBL_EXAM_COURSE}.content_id = ?`],
    parameters: [content_id],
  }

  const result = await deleteExamService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while deleting exam");
  }

  return res.status(200).json({ success: true, message: "Exam list deleted successfully" });

});

module.exports = {
  get_examList,
  post_examList,
  update_examList,
  del_examList,
};


const setExamAttributes = () => {
  return (
    {
      required: [
        "exam_url",
        "show_on_home",
        "is_right_section",
        "exam_name",
        "category",
      ],
      attributes: [
        "category",
        "coursename",
        "exam_name",
        "exam_url",
        "show_on_home",
        "is_right_section",
        "exam_logo",
        "title",
        "meta_key",
        "meta_des",
        "content",
        "application_start_date",
        "application_end_date",
        "exam_start_date",
        "exam_end_date",
        "admit_card_date",
        "exam_full_name",
        "level",
        "counselling_start_date",
        "counselling_end_date",
        "counselling_start_date_online",
        "counselling_end_date_online",
        "counselling_start_date_offline",
        "counselling_end_date_offline",
        "app_start_date_online",
        "app_end_date_online",
        "app_start_date_offline",
        "app_end_date_offline",
        "exam_start_date_online",
        "exam_end_date_online",
        "exam_start_date_offline",
        "exam_end_date_offline",
        "result_date",
        "exam_level",
        "tab1_title",
        "tab1_content",
        "tab2_title",
        "tab2_content",
        "tab3_title",
        "tab3_content",
        "tab4_title",
        "tab4_content",
        "tab5_title",
        "tab5_content",
        "tab6_title",
        "tab6_content",
        "tab7_title",
        "tab7_content",
        "tab8_title",
        "tab8_content",
        "tab9_title",
        "tab9_content",
        // "priority",
      ]
    }
  )
}