const { urlExist, postUrlExist } = require("../../../utils/url_exist");
const { errorMsgEmpty } = require("../../../utils/errors_message");
const { CATEGORY } = require("../../../models/tables");
const { getCategoryService, addNewCategoryService, deleteCategoryService, updateCategoryService } = require("../../../services/category.service");
const { areAllKeysPresent } = require("../../../utils/common_utils");
const tryCatch = require("../../../utils/tryCatch");
const updateColumn = require("../../../utils/updateColumn");

// @desc get all the category
// @api POST /api/v1/admin/category
// @access Private
const get_category = tryCatch(async (req, res) => {

  const { search } = req.query;
  const queryMap = {};
  queryMap.attributes = [`${CATEGORY}.*`];

  if(search){
    queryMap.conditions = [
      `${CATEGORY}.cat_name LIKE '%${search}%'`,
      `${CATEGORY}.active = 'Y'`,
    ];
    queryMap.parameters = [`${search}%`];
  }

  const result = await getCategoryService(queryMap);
  if (result instanceof Error || result.length === 0) {
    errorMsgEmpty("No data found", res);
  }

  return res.status(200).json({
    success: true,
    data: result,
  });

});

// @desc post new category
// @api POST /api/v1/admin/category
// @access Private ADMIN
const post_category = tryCatch(async (req, res) => {

  const requiredFields = ["cat_name", "cat_url", "isCourse", "isExam", "iscareer", "cat_image", "show_on_home", "is_live", "priority"];
  if(!areAllKeysPresent(req.body, requiredFields)){
    throw new Error("All fields are required");
  }

  req.body.cat_url && await postUrlExist(CATEGORY, "cat_url", req.body.cat_url);

  const queryMap = {};
  queryMap.attributes = requiredFields;
  queryMap.parameters = requiredFields.map((field) => req.body[field]);

  const result = await addNewCategoryService(queryMap);
  if (result instanceof Error) {
    res.statusCode = 400;
    throw result;
  }

  return res.status(200).json({
    success: true,
    message: "Category Added!",
  });

});

// @desc update category based on cat_id
// @api PUT /api/v1/admin/category
// @access Private ADMIN
const update_category = tryCatch(async (req, res) => {

  const { cat_id, update } = req.body;
  if (!cat_id || !update) throw new Error(`category ID and Update are required`);

  update.cat_url && await urlExist(CATEGORY, "cat_id", "cat_url", update.cat_url, cat_id);

  const { column, value }= updateColumn(update);

  const queryMap = {};
  queryMap.update = column;
  queryMap.conditions = [`${CATEGORY}.cat_id = ?`];
  queryMap.parameters = [...value, cat_id];

  const result = await updateCategoryService(queryMap);
  if (result instanceof Error) {
    res.statusCode = 400;
    throw result;
  }

  return res.status(200).json({
    success: true,
    message: "Category Updated!",
  });

});

// @desc Delete category based on cat_id
// @api DELETE /api/v1/admin/category
// @access Private ADMIN
const del_category = tryCatch(async (req, res) => {

  const { cat_id } = req.body;
  if (!cat_id) throw new Error("no cat_id given");

  const queryMap = {};
  queryMap.conditions = [`${CATEGORY}.cat_id = ?`];
  queryMap.parameters = [cat_id];

  const result = await deleteCategoryService(queryMap);
  if (result instanceof Error) {
    res.statusCode = 400;
    throw result;
  }

  return res.status(200).json({
    success: true,
    message: "Category Deleted!",
  });

});

module.exports = {
  get_category,
  post_category,
  update_category,
  del_category,
};
