const { dbQuery } = require("../../../config/db.config");
const tryCatch = require("../../../utils/tryCatch");
const updateColumn = require("../../../utils/updateColumn");
const { errorMsgEmpty } = require("../../../utils/errors_message");
const { INSTITUTE_TYPE } = require("../../../models/tables");
const {
  getCollegeTypeService,
  addNewCollegeTypeService,
  deleteCollegeTypeService,
  updateCollegeTypeService,
} = require("../../../services/master/collegeType.service");
const { areAllKeysPresent } = require("../../../utils/common_utils");

// @desc get all college type
// @route GET /api/v1/admin/collegeType
// @access Private
const get_collegeType = tryCatch(async (req, res) => {
  const queryMap = {
    attributes: [`${INSTITUTE_TYPE}.*`],
  };

  const result = await getCollegeTypeService(queryMap);
  if (result instanceof Error) {
    errorMsgEmpty("No data found", res);
  }

  return res.status(200).json({ success: true, data: result });
});

// @desc post new college type
// @route POST /api/v1/admin/collegeType
// @access Private ADMIN
const post_collegeType = tryCatch(async (req, res) => {
 
  const requiredFields = ["name", "priority"];
  if (!areAllKeysPresent(req.body, requiredFields)){
    throw new Error("All fields are required");
  }

  const queryMap = {};
  queryMap.attributes = requiredFields;
  queryMap.parameters = requiredFields.map((field) => req.body[field]);

  const result = await addNewCollegeTypeService(queryMap);
  if (result instanceof Error) {
    res.statusCode = 400;
    throw result;
  }

  return res.status(200).json({
    success: true,
    message: "College Type Added!",
  });

});

// @desc update college type
// @route PUT /api/v1/admin/collegeType
// @access Private ADMIN
const update_collegeType = tryCatch(async (req, res) => {

  const { college_id, update } = req.body;
  if (!college_id || !update){
    throw new Error(`Please provide valid id and update`);
  }

  const { column, value } = updateColumn(update);

  const queryMap = {
    update: column,
    conditions: [`${INSTITUTE_TYPE}.id = ?`],
    parameters: [...value, college_id],
  };

  const result = await updateCollegeTypeService(queryMap);
  if (result instanceof Error) {
    res.statusCode = 400;
    throw result;
  }

  return res.status(200).json({
    success: true,
    message: "College Type Updated",
  });
});

// @desc delete college type
// @route DELETE /api/v1/admin/collegeType
// @access Private ADMIN
const del_collegeType = tryCatch(async (req, res) => {

  const { college_id } = req.body;
  if (!college_id){
    throw new Error(`Please provide valid id that exist`);
  }

  const queryMap = {
    conditions: [`${INSTITUTE_TYPE}.id = ?`],
    parameters: [college_id],
  }

  const result = await deleteCollegeTypeService(queryMap);
  if (result instanceof Error) {
    res.statusCode = 400;
    throw result;
  };

  return res.status(200).json({
    success: true,
    message: "College Type Deleted!",
  });
});

module.exports = {
  get_collegeType,
  post_collegeType,
  update_collegeType,
  del_collegeType,
};
