const { errorMsgEmpty } = require("../../../utils/errors_message");
const { COMPANY_PLACEMENT } = require("../../../models/tables");
const {
  getCompanyPlacementPaginationService,
  addNewCompanyPlacementService,
  updateCompanyPlacementService,
  deleteCompanyPlacementService,
} = require("../../../services/master/companyPlacement.service");
const tryCatch = require("../../../utils/tryCatch");
const updateColumn = require("../../../utils/updateColumn");

// @desc get 50 companies at a time
// @api GET /api/v1/admin/company-placement
// @access Private
const get_companyPlacement = tryCatch(async (req, res) => {

  const { page = 1, limit = 50, search } = req.query;
  if(Number(page) < 1) throw new Error("Page number is required and should be greater than 0");
  const offset = (page - 1) * limit;

  const queryMap = {
    attributes: [`${COMPANY_PLACEMENT}.*`],
    additional: [`ORDER BY ${COMPANY_PLACEMENT}.insert_date DESC`, `LIMIT ${limit} OFFSET ${offset}`],
    requirePagination: true,
  }
  if(search){
    queryMap.conditions = ["company_name LIKE ?"];
    queryMap.parameters = [`${search}%`];
  }

  const { result, total } = await getCompanyPlacementPaginationService(queryMap);
  if(result instanceof Error) errorMsgEmpty("No Company Placement found", res);

  return res.status(200).json({
    success: true,
    message: "Company Placement fetched successfully",
    data: result,
    records:total,
  });

});

// @desc post company placement
// @api POST /api/v1/admin/company-placement
// @access Private ADMIN
const post_companyPlacement = tryCatch(async (req, res) => {
 
  const { company_name, company_logo, company_des, priority, active } = req.body;
  if(!company_name) throw new Error("Company name is required");

  const queryMap = {
    attributes: ["company_name", "company_logo", "company_des", "priority", "insert_date", "active"],
    parameters: [company_name, company_logo, company_des, priority, new Date(), 'Y'],
  };

  const result = await addNewCompanyPlacementService(queryMap);
  if(result instanceof Error) throw new Error("Error! cant add company placement");

  return res.status(200).json({
    success: true,
    message: "Company placement added successfully",
  });

});

// @desc update company placement
// @api PUT /api/v1/admin/company-placement
// @access Private ADMIN
const update_companyPlacement = tryCatch(async (req, res) => {

  const { com_id, update } = req.body;
  if (!com_id || !update) throw new Error("com_id and update are required");

  const { column, value } = updateColumn(update);

  const queryMap = {
    update: [column, "update_date = NOW()"].join(", "),
    conditions: ["com_id = ?"],
    parameters: [...value, com_id],
  }

  const result = await updateCompanyPlacementService(queryMap);
  if(result instanceof Error) throw new Error("Error! cant update company placement");

  return res.status(200).json({
    success: true,
    message: "Company placement updated successfully",
  });

});

// @desc delete company placement
// @api DELETE /api/v1/admin/company-placement
// @access Private ADMIN
const del_companyPlacement = tryCatch(async (req, res) => {

  const { com_id } = req.body;
  if (!com_id) throw new Error("com_id required!");

  const queryMap = {
    conditions: ["com_id = ?"],
    parameters: [com_id],
  }

  const result = await deleteCompanyPlacementService(queryMap);
  if(result instanceof Error) throw new Error("Error! cant delete company placement");

  return res.status(200).json({
    success: true,
    message: "Company placement deleted successfully",
  });

});
module.exports = {
  get_companyPlacement,
  post_companyPlacement,
  update_companyPlacement,
  del_companyPlacement,
};
