const { errorMsgEmpty } = require("../../../utils/errors_message");
const { COST_OF_STUDY } = require("../../../models/tables");
const {
  getCostOfStudyPaginationService,
  addNewCostOfStudyService,
  updateCostOfStudyService,
  deleteCostOfStudyService,
} = require("../../../services/master/costOfStudy.service");
const tryCatch = require("../../../utils/tryCatch");
const updateColumn = require("../../../utils/updateColumn");

const table = COST_OF_STUDY;

// @desc get all Cost of study
// @api GET /api/admin/cost-of-study
// @access Private
const get_costOfStudy = tryCatch(async (req, res) => {

  const { page = 1, limit = 50, search } = req.query;
  if(Number(page) < 1) throw new Error("Page number is required and should be greater than 0");
  const offset = Number((page - 1) * limit);

  const queryMap = {
    attributes: [`${COST_OF_STUDY}.*`],
    additional: [`ORDER BY ${COST_OF_STUDY}.priority ASC`,`LIMIT ${Number(limit)} OFFSET ${offset}`],
  }
  if(search){
    queryMap.conditions = ["name LIKE ?"];
    queryMap.parameters = [`${search}%`];
  }

  const { result, total } = await getCostOfStudyPaginationService(queryMap);
  if(result instanceof Error) errorMsgEmpty("No Cost of Study found", res);

  return res.status(200).json({
    success: true,
    message: "Cost of Study fetched successfully",
    data: result,
    records:total,
  });

});

// @desc post new Cost of study
// @api POST /api/admin/cost-of-study
// @access Private ADMIN
const post_costOfStudy = tryCatch(async (req, res) => {

  const { name, priority } = req.body;
  if (!name) throw new Error("name is required");

  const queryMap = {
    attributes: ["name", "priority"],
    parameters: [name, (Number(priority) || 0)],
  }

  const result = await addNewCostOfStudyService(queryMap);
  if(result instanceof Error) throw new Error("Error! Not added");

  return res.status(200).json({
    success: true,
    message: "Cost of Study Added!",
  });

});

// @desc update Cost of study
// @api PUT /api/admin/cost-of-study
// @access Private ADMIN
const update_costOfStudy = tryCatch(async (req, res) => {

  const { id, update } = req.body;
  if (!id || !update) throw new Error("id and update are required");

  const { column, value } = updateColumn(update);

  const queryMap = {
    update: column,
    conditions: ["id = ?"],
    parameters: [...value, id],
  }

  const result = await updateCostOfStudyService(queryMap);
  if(result instanceof Error) throw new Error("Error! Not Updated");

  return res.status(200).json({
    success: true,
    message: "Cost of Study Updated!",
  });
  
});

// @desc delete Cost of study
// @api DELETE /api/admin/cost-of-study
// @access Private ADMIN
const del_costOfStudy = tryCatch(async (req, res) => {

  const { id } = req.body;
  if (!id) throw new Error("id is required");

  const queryMap = {
    conditions: ["id = ?"],
    parameters: [id],
  }

  const result = await deleteCostOfStudyService(queryMap);
  if(result instanceof Error) throw new Error("Error! Not Deleted");

  return res.status(200).json({
    success: true,
    message: "Cost of Study Deleted!",
  });
});

module.exports = {
  get_costOfStudy,
  post_costOfStudy,
  update_costOfStudy,
  del_costOfStudy,
};
