const tryCatch = require("../../../utils/tryCatch");
const updateColumn = require("../../../utils/updateColumn");
const { postUrlExist, urlExist } = require("../../../utils/url_exist");
const { COURSE, CATEGORY } = require("../../../models/tables");
const { getCoursePaginationService, addNewCourseService, updateCourseService, deleteCourseService } = require("../../../services/master/course.service");
const { areAllKeysPresent } = require("../../../utils/common_utils");

// @desc get all Course details 50 limit
// @api GET /api/v1/admin/course
// @access Private
const get_course = tryCatch(async (req, res) => {
  
  const { page = 1, limit = 50, search, category } = req.query;
  if(Number(page) < 1) throw new Error("Page number is required and should be greater than 0");
  const offset = Number((page - 1) * limit);

  const queryMap = {
    attributes: [`${COURSE}.*`, `${CATEGORY}.cat_name`],
    joinTbl:[`LEFT JOIN ${CATEGORY} ON ${CATEGORY}.cat_id = ${COURSE}.cat_id`],
    conditions:[],
    parameters:[],
    additional:[`ORDER BY ${COURSE}.priority ASC`, `LIMIT ${Number(limit)} OFFSET ${offset}`]
  };

  if(search){
    queryMap.conditions.push(`${COURSE}.course_name LIKE ?`);
    queryMap.parameters.push(`%${search}%`);
  }

  if(category){
    queryMap.conditions.push(`${CATEGORY}.cat_name LIKE ?`);
    queryMap.parameters.push(`%${category}%`);
  }

  const { result, total } = await getCoursePaginationService(queryMap);
  
  return res.status(200).json({
    success:true,
    data:result,
    records:total
  })

});

// @desc post course
// @api POST /api/v1/admin/course
// @access Private ADMIN
const post_course = tryCatch(async (req, res) => {
  const courseAttributes = getCourseAttributes().attributes;
  const requiredFields = getCourseAttributes().requiredFields;
  if (!areAllKeysPresent(req.body, requiredFields)){
    throw new Error("All fields are required");
  }

  req.body.course_url && await postUrlExist(COURSE, "course_url", req.body.course_url);

  const queryMap = {
    attributes: courseAttributes.concat("priority"),
    parameters: courseAttributes.map((field) => (req.body[field] || "")).concat((req.body.priority || 20)),
  }

  const result = await addNewCourseService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while adding new course");
  }

  return res.status(200).json({
    success:true,
    message: "New course added"
  })
});

// @desc update course
// @api PUT /api/v1/admin/course
// @access Private ADMIN
const update_course = tryCatch(async (req, res) => {

  const { course_id, update } = req.body;
  if (!course_id || !update) throw new Error("course_id and update required");

  update.course_url && (await urlExist( COURSE, "course_id", "course_url", update.course_url, course_id ));

  const { column, value } = updateColumn(update);

  const queryMap = {
    update: column,
    conditions: [`${COURSE}.course_id = ?`],
    parameters: [...value, course_id]
  };

  const result = await updateCourseService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while updating the course");
  }

  return res.status(200).json({
    success: true,
    message:"Course has been updated"
  })

});

// @desc delete course
// @api DELETE /api/v1/admin/course
// @access Private ADMIN
const del_course = tryCatch(async (req, res) => {
  
  const { course_id } = req.body;
  if (!course_id) throw new Error("course_id required");

  const queryMap = {
    conditions: [`${COURSE}.course_id = ?`],
    parameters: [course_id],
  };

  const result = await deleteCourseService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while deleting the course");
  }

  return res.status(200).json({
    success: true,
    message: "Course deleted successfully"
  })
});

const getCourseAttributes = () => {
  return {
    attributes: ['cat_id', 'course_name', 'course_fullname', 'short_name', 'course_duration', 'degree_type', 'course_url', 'show_in_dropdown', 'show_on_home'],
    requiredFields: ['cat_id', 'course_name', 'course_url']
  }
}

module.exports = {
  get_course,
  post_course,
  update_course,
  del_course,
};
