const tryCatch = require("../../../utils/tryCatch");
const updateColumn = require("../../../utils/updateColumn");
const { COURSE_DELIVERY } = require("../../../models/tables");
const { getCourseDeliveryPaginationService, addNewCourseDeliveryService, updateCourseDeliveryService, deleteCourseDeliveryService } = require("../../../services/master/courseDelivery.service");

// @desc get all course delivery
// @api GET /api/v1/admin/course-delivery
// @access Private
const get_courseDelivery = tryCatch(async (req, res) => {
  
  const { page = 1, limit = 50, search } = req.query;
  if(Number(page) < 1) throw new Error("Page number is required and should be greater than 0");
  const offset = Number((page - 1) * limit);

  const queryMap = {
    attributes: [`${COURSE_DELIVERY}.*`],
    conditions:[],
    parameters:[],
    additional:[`ORDER BY ${COURSE_DELIVERY}.priority ASC`, `LIMIT ${Number(limit) || 10} OFFSET ${offset}`]
  };

  if(search){
    queryMap.conditions.push(`${COURSE_DELIVERY}.name LIKE ?`);
    queryMap.parameters.push(`%${search}%`);
  }

  const { result, total } = await getCourseDeliveryPaginationService(queryMap);
  
  return res.status(200).json({
    success:true,
    data:result,
    records:total
  })
  
});

// @desc post new course delivery
// @api POST /api/v1/admin/course-delivery
// @access Private ADMIN
const post_courseDelivery = tryCatch(async (req, res) => {

  const { name, priority } = req.body;
  if (!name) throw new Error("Name is required");
 
  const queryMap = {
    attributes: ['name', 'priority'],
    parameters: [name, (priority || 10)]
  }

  const result = await addNewCourseDeliveryService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while adding new course delivery");
  }

  return res.status(200).json({
    success:true,
    message: "New course delivery added"
  })

});

// @desc update course delivery
// @api PUT /api/v1/admin/course-delivery
// @access Private ADMIN
const update_courseDelivery = tryCatch(async (req, res) => {

  const { id, update } = req.body;
  if (!id || !update) throw new Error("id and update required");

  const { column, value } = updateColumn(update);

  const queryMap = {
    update: column,
    conditions: [`${COURSE_DELIVERY}.id = ?`],
    parameters: [...value, id]
  };

  const result = await updateCourseDeliveryService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while updating the course delivery");
  }

  return res.status(200).json({
    success: true,
    message:"Course Delivery has been updated"
  })

});

// @desc delete course delivery
// @api DELETE /api/v1/admin/course-delivery
// @access Private ADMIN
const del_courseDelivery = tryCatch(async (req, res) => {
  
  const { id } = req.body;
  if (!(typeof id === "number") && !id) throw new Error("id required");

  const queryMap = {
    conditions: [`${COURSE_DELIVERY}.id = ?`],
    parameters: [id],
  };

  const result = await deleteCourseDeliveryService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while deleting the course delivery");
  }

  return res.status(200).json({
    success: true,
    message: "Course delivery deleted successfully"
  })

});

module.exports = {
  get_courseDelivery,
  post_courseDelivery,
  update_courseDelivery,
  del_courseDelivery,
};
