const { dbQuery } = require("../../../config/db.config");
const tryCatch = require("../../../utils/tryCatch");
const updateColumn = require("../../../utils/updateColumn");
const { COURSELEVEL } = require("../../../models/tables");
const { getCourseLevelPaginationService, addNewCourseLevelService, updateCourseLevelService, deleteCourseLevelService } = require("../../../services/master/courseLevel.service");

const table = COURSELEVEL;

// @desc get all the courselevel
// @api GET /api/v1/admin/course-level
// @access Private
const get_level = tryCatch(async (req, res) => {
  
  const { page = 1, limit = 50, search } = req.query;
  if(Number(page) < 1) throw new Error("Page number is required and should be greater than 0");
  const offset = Number((page - 1) * limit);

  const queryMap = {
    attributes: [`${COURSELEVEL}.*`],
    conditions:[],
    parameters:[],
    additional:[`ORDER BY ${COURSELEVEL}.priority ASC`, `LIMIT ${Number(limit) || 10} OFFSET ${offset}`]
  };

  if(search){
    queryMap.conditions.push(`${COURSELEVEL}.name LIKE ?`);
    queryMap.parameters.push(`%${search}%`);
  }

  const { result, total } = await getCourseLevelPaginationService(queryMap);
  
  return res.status(200).json({
    success:true,
    data:result,
    records:total
  })

});

// @desc post new courselevel
// @api POST /api/v1/admin/course-level
// @access Private ADMIN
const post_level = tryCatch(async (req, res) => {
 
  const { name, priority } = req.body;
  if (!name) throw new Error("Name is required");
 
  const queryMap = {
    attributes: ['name', 'priority'],
    parameters: [name, (priority || 10)]
  }

  const result = await addNewCourseLevelService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while adding new course level");
  }

  return res.status(200).json({
    success:true,
    message: "New course level added"
  })

});

// @desc update courselevel
// @api POST /api/v1/admin/course-level
// @access Private ADMIN
const update_level = tryCatch(async (req, res) => {
 
  const { id, update } = req.body;
  if (!id || !update) throw new Error("id and update required");

  const { column, value } = updateColumn(update);

  const queryMap = {
    update: column,
    conditions: [`${COURSELEVEL}.id = ?`],
    parameters: [...value, id]
  };

  const result = await updateCourseLevelService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while updating the course delivery");
  }

  return res.status(200).json({
    success: true,
    message:"Course level has been updated"
  })
  
});

// @desc delete courselevel
// @api POST /api/v1/admin/course-level
// @access Private ADMIN
const del_level = tryCatch(async (req, res) => {

  const { id } = req.body;
  if (!id) throw new Error("id required");

  const queryMap = {
    conditions: [`${COURSELEVEL}.id = ?`],
    parameters: [id],
  };

  const result = await deleteCourseLevelService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while deleting the course level");
  }

  return res.status(200).json({
    success: true,
    message: "Course level deleted successfully"
  })

});

module.exports = {
  get_level,
  post_level,
  update_level,
  del_level,
};
