const tryCatch = require("../../../utils/tryCatch");
const updateColumn = require("../../../utils/updateColumn");
const { EXAM_LEVEL_TYPE } = require("../../../models/tables");
const { getExamLevelPaginationService, addNewExamLevelService, updateExamLevelService, deleteExamLevelService } = require("../../../services/master/examLevel.service");

// @desc get all exam level
// @api GET /api/v1/admin/exam-level
// @access Private
const get_examLevel = tryCatch(async (req, res) => {
 
  const { page = 1, limit = 50, search } = req.query;
  if(Number(page) < 1) throw new Error("Page number is required and should be greater than 0");
  const offset = Number((page - 1) * limit);

  const queryMap = {
    attributes: [`${EXAM_LEVEL_TYPE}.*`],
    conditions:[],
    parameters:[],
    additional:[`ORDER BY ${EXAM_LEVEL_TYPE}.priority ASC`, `LIMIT ${Number(limit) || 10} OFFSET ${offset}`]
  };

  if(search){
    queryMap.conditions.push(`${EXAM_LEVEL_TYPE}.name LIKE ?`);
    queryMap.parameters.push(`%${search}%`);
  }

  const { result, total } = await getExamLevelPaginationService(queryMap);
  
  return res.status(200).json({
    success:true,
    data:result,
    records:total
  })

});

// @desc post exam level
// @api POST /api/v1/admin/exam-level
// @access Private ADMIN
const post_examLevel = tryCatch(async (req, res) => {
 
  const { name, priority, active } = req.body;
  if (!name) throw new Error("Name is required");
 
  const queryMap = {
    attributes: ['name', 'priority', 'active'],
    parameters: [name, (priority || 10), (active || 'Y')]
  }

  const result = await addNewExamLevelService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while adding new exam level");
  }

  return res.status(200).json({
    success:true,
    message: "New exam level added"
  })

});

// @desc update exam level
// @api PUT /api/v1/admin/exam-level
// @access Private ADMIN
const update_examLevel = tryCatch(async (req, res) => {

  const { id, update } = req.body;
  if (!id || !update) throw new Error("id and update required");

  const { column, value } = updateColumn(update);

  const queryMap = {
    update: column,
    conditions: [`${EXAM_LEVEL_TYPE}.id = ?`],
    parameters: [...value, id]
  };

  const result = await updateExamLevelService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while updating the exam level");
  }

  return res.status(200).json({
    success: true,
    message:"Exam level has been updated"
  })

});

// @desc delete exam level
// @api DELETE /api/v1/admin/exam-level
// @access Private ADMIN
const del_examLevel = tryCatch(async (req, res) => {
  
  const { id } = req.body;
  if (!id) throw new Error("id required");

  const queryMap = {
    conditions: [`${EXAM_LEVEL_TYPE}.id = ?`],
    parameters: [id],
  };

  const result = await deleteExamLevelService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while deleting the exam level");
  }

  return res.status(200).json({
    success: true,
    message: "Exam level deleted successfully"
  })

});

module.exports = {
  get_examLevel,
  post_examLevel,
  update_examLevel,
  del_examLevel,
};
