const { dbQuery } = require("../../../config/db.config");
const tryCatch = require("../../../utils/tryCatch");
const updateColumn = require("../../../utils/updateColumn");
const { SCHOOL_FACILITY } = require("../../../models/tables");
const { getSchoolFacilityPaginationService, addNewSchoolFacilityService, updateSchoolFacilityService, deleteSchoolFacilityService } = require("../../../services/master/schoolFacility.service");
const { areAllKeysPresent } = require("../../../utils/common_utils")

const table = SCHOOL_FACILITY;

// @desc get all School Facilities
// @api GET /api/v1/admin/schoolfacility
// @access Private
const get_schoolFacility = tryCatch(async (req, res) => {
 
    const { page = 1, limit = 50, search } = req.query;
    if(Number(page) < 1) throw new Error("Page number is required and should be greater than 0");
    const offset = (page - 1) * limit;
  
    const queryMap = {
      attributes: [`${SCHOOL_FACILITY}.*`],
      additional: [`ORDER BY ${SCHOOL_FACILITY}.cat_id DESC`, `LIMIT ${Number(limit) || 10} OFFSET ${offset}`],
    }

    if(search){
      queryMap.conditions = [`${SCHOOL_FACILITY}.cat_name LIKE ?`];
      queryMap.parameters = [`%${search}%`];
    }
  
    const { result, total } = await getSchoolFacilityPaginationService(queryMap);
  
    return res.status(200).json({
      success: true,
      data: result,
      records:total,
    });
  
});

// @desc post new School Facility
// @api POST /api/v1/admin/schoolfacility
// @access Private ADMIN
const post_schoolFacility = tryCatch(async (req, res) => {
   
  const { cat_name, icon, isSchool, isCollege, priority } = req.body;
 
  const requiredFields = ["cat_name", "isSchool", "isCollege"];
  if(!areAllKeysPresent(req.body, requiredFields)){
    res.statusCode = 400;
    throw new Error("some required fields are missing")
  }

  const queryMap = {
    attributes: ["cat_name", "isSchool", "isCollege", "icon", "priority"],
    parameters: [cat_name, isSchool, isCollege, icon, priority],
  };

  const result = await addNewSchoolFacilityService(queryMap);
  if(result instanceof Error) throw new Error("Error! cant add school facility");

  return res.status(200).json({
    success: true,
    message: "School Facility added successfully",
  });


});

// @desc update School Facility
// @api PUT /api/v1/admin/schoolfacility
// @access Private ADMIN
const update_schoolFacility = tryCatch(async (req, res) => {

  const { cat_id, update } = req.body;
  if (!cat_id || !update) throw new Error("cat_id and update are required");

  if(!update.icon) update.icon = ""; // CANT NOT BE null So "" instead

  const { column, value } = updateColumn(update);

  const queryMap = {
    update: column,
    conditions: [`${SCHOOL_FACILITY}.cat_id = ?`],
    parameters: [...value, cat_id],
  }

  const result = await updateSchoolFacilityService(queryMap);
  if(result instanceof Error) throw new Error("Error while updating school facility");

  return res.status(200).json({
    success: true,
    message: "School Facility updated",
  });
 
});

// @desc delete School Facility
// @api DELETE /api/v1/admin/schoolfacility
// @access Private ADMIN
const del_schoolFacility = tryCatch(async (req, res) => {

  const { cat_id } = req.body;
  if (!Number(cat_id)) throw new Error("cat_id required");

  const queryMap = {
    conditions: [`${SCHOOL_FACILITY}.cat_id = ?`],
    parameters: [cat_id],
  };

  const result = await deleteSchoolFacilityService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while deleting the School Facility");
  }

  return res.status(200).json({
    success: true,
    message: "School Facility deleted successfully"
  })

});

module.exports = {
  get_schoolFacility,
  post_schoolFacility,
  update_schoolFacility,
  del_schoolFacility,
};
