const tryCatch = require("../../../utils/tryCatch");
const updateColumn = require("../../../utils/updateColumn");
const { SCHOOLTYPE } = require("../../../models/tables");
const {
  getSchoolTypeService,
  deleteSchoolTypeService,
  addNewSchoolTypeService,
  updateSchoolTypeService,
  getSchoolTypePaginationService,
} = require("../../../services/master/schoolType.service");
const { areAllKeysPresent } = require("../../../utils/common_utils");


// @desc get all School Types
// @api GET /api/v1/admin/schooltype
// @access Private
const get_schoolType = tryCatch(async (req, res) => {

  const { search, limit = 20, page = 1 } = req.query;
  if(Number(page) < 1) throw new Error("Page number is required and should be greater than 0");
  const offset = (page - 1) * limit;

  const queryMap = {
    attributes: [`${SCHOOLTYPE}.*`],
    conditions: [],
    parameters: [],
    additional: [`LIMIT ${Number(limit) || 10} OFFSET ${offset}`]
  };

  if(search){
    queryMap.conditions.push(`${SCHOOLTYPE}.name LIKE ?`);
    queryMap.parameters.push(`%${search}%`);
  }

  const { result, total } = await getSchoolTypePaginationService(queryMap);

  return res.status(200).json({
    success: true,
    data: result,
    records: total,
  });
});

// @desc post new School Type
// @api POST /api/v1/admin/schooltype
// @access Private ADMIN
const post_schoolType = tryCatch(async (req, res) => {
  const requiredFields = ["name", "list_url", "details_url"];
  if (!areAllKeysPresent(req.body, requiredFields)) {
    res.statusCode = 400;
    throw new Error("Some fields are missing");
  }

  const queryMap = {
    attributes: [...requiredFields, "priority"],
    parameters: [
      ...requiredFields.map((field) => req.body[field]),
      req.body.priority || 10,
    ],
  };

  const result = await addNewSchoolTypeService(queryMap);
  if (result instanceof Error) {
    res.statusCode = 400;
    throw new Error("Error while creating new school type");
  }

  return res.status(200).json({
    success: true,
    message: "Successfully created new school type",
  });
});

// @desc update School Type
// @api PUT /api/v1/admin/schooltype
// @access Private ADMIN
const update_schoolType = tryCatch(async (req, res) => {
  const { school_id, update } = req.body;
  if (!Number(school_id) || !update) {
    res.statusCode = 400;
    throw new Error("school id or update is missing");
  }

  const { column, value } = updateColumn(update);

  const queryMap = {
    update: column,
    conditions: [`${SCHOOLTYPE}.school_id = ?`],
    parameters: [...value, school_id],
  };

  const result = await updateSchoolTypeService(queryMap);
  if (result instanceof Error) {
    res.statusCode = 400;
    throw new Error("Error while updating school type");
  }

  return res.status(200).json({
    success: true,
    message: "School type updated successfully",
  });
});

// @desc delete School Type
// @api DELETE /api/v1/admin/schooltype
// @access Private ADMIN
const del_schoolType = tryCatch(async (req, res) => {
  const { school_id } = req.body;
  if (!Number(school_id)) {
    res.statusCode = 400;
    throw new Error("No school id provided");
  }

  const queryMap = {
    conditions: [`${SCHOOLTYPE}.school_id = ?`],
    parameters: [school_id],
  };

  const result = await deleteSchoolTypeService(queryMap);
  if (result instanceof Error) {
    res.statusCode = 400;
    throw new Error("Error while deleting the school type");
  }

  return res.status(200).json({
    success: true,
    message: "Successfully deleted school type",
  });
});

module.exports = {
  get_schoolType,
  post_schoolType,
  update_schoolType,
  del_schoolType,
};
