const tryCatch = require("../../../utils/tryCatch");
const updateColumn = require("../../../utils/updateColumn");
const { areAllKeysPresent } = require("../../../utils/common_utils");
const { urlExist, postUrlExist } = require("../../../utils/url_exist");
const { SPECIALIZATION, COURSE, CATEGORY } = require("../../../models/tables");
const {
  getSpecializationPaginationService,
  addNewSpecializationService,
  updateSpecializationService,
  deleteSpecializationService,
} = require("../../../services/master/specialization.service");

// @desc get all the specialization
// @api GET /api/v1/admin/specialization
// @access Private
const get_specialization = tryCatch(async (req, res) => {
  const { page = 1, limit = 50, search, category } = req.query;
  if (Number(page) < 1)
    throw new Error("Page number is required and should be greater than 0");
  const offset = Number((page - 1) * limit);

  const queryMap = {
    attributes: [
      `${SPECIALIZATION}.*`,
      `${COURSE}.course_name`,
      `${CATEGORY}.cat_name`,
    ],
    conditions: [],
    parameters: [],
    joinTbl: [
      `LEFT JOIN ${COURSE} ON ${COURSE}.course_id = ${SPECIALIZATION}.course_id`,
      `LEFT JOIN ${CATEGORY} ON ${CATEGORY}.cat_id = ${SPECIALIZATION}.cat_id`,
    ],
    additional: [
      `ORDER BY ${SPECIALIZATION}.insert_date DESC`,
      `LIMIT ${Number(limit) || 10} OFFSET ${offset}`,
    ],
  };

  if (search) {
    queryMap.conditions.push(`${SPECIALIZATION}.spce_name LIKE ?`);
    queryMap.parameters.push(`%${search}%`);
  }

  if (category) {
    queryMap.conditions.push(`${CATEGORY}.cat_name LIKE ?`);
    queryMap.parameters.push(`%${category}%`);
  }

  const { result, total } = await getSpecializationPaginationService(queryMap);

  return res.status(200).json({
    success: true,
    data: result,
    records: total,
  });
});

// @desc post specialization
// @api POST /api/v1/admin/specialization
// @access Private ADMIN
const post_specialization = tryCatch(async (req, res) => {
  
  const requiredFields = ["course_id", "cat_id", "spce_name", "specialization_url", "show_in_dropdown"];
  if(!areAllKeysPresent(req.body, requiredFields)) throw new Error(`Some required fields are missing!`);

  req.body.specialization_url && (await postUrlExist(SPECIALIZATION, "specialization_url", req.body.specialization_url));

  const queryMap = {
    attributes: [...fieldsForNewSpecialization(), "insert_date", "active", "priority"],
    parameters: [...fieldsForNewSpecialization().map(field => req.body[field]), new Date(), (req.body.active || 'Y'), (req.body.priority || 10)]
  };


  
  const result = await addNewSpecializationService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while adding new specialization");
  }

  return res.status(200).json({
    success:true,
    message: "New specialization added"
  });
  
});

// @desc update specialization
// @api PUT /api/v1/admin/specialization
// @access Private ADMIN
const update_specialization = tryCatch(async (req, res) => {

  const { spec_id, update } = req.body;
  if (!spec_id || !update) throw new Error("spec_id and update required");

  update.specialization_url &&
  (await urlExist(
    SPECIALIZATION,
    "spec_id",
    "specialization_url",
    update.specialization_url,
    spec_id
  ));

  const { column, value } = updateColumn(update);

  const queryMap = {
    update: [column, "updated_date = NOW()"].join(", "),
    conditions: [`${SPECIALIZATION}.spec_id = ?`],
    parameters: [...value, spec_id]
  };

  const result = await updateSpecializationService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while updating the exam level");
  }

  return res.status(200).json({
    success: true,
    message:"Exam level has been updated"
  })

});

// @desc delete specialization
// @api DELETE /api/v1/admin/specialization
// @access Private ADMIN
const del_specialization = tryCatch(async (req, res) => {

  const { spec_id } = req.body;
  if (!spec_id) throw new Error("spec_id required");

  const queryMap = {
    conditions: [`${SPECIALIZATION}.spec_id = ?`],
    parameters: [spec_id],
  };

  const result = await deleteSpecializationService(queryMap);
  if(result instanceof Error){
    res.statusCode = 400;
    throw new Error("Error while deleting the exam level");
  }

  return res.status(200).json({
    success: true,
    message: "Exam level deleted successfully"
  })

});

module.exports = {
  get_specialization,
  post_specialization,
  update_specialization,
  del_specialization,
};

const fieldsForNewSpecialization = () => {
  return [
    "course_id",
    "course_url",
    "cat_id",
    "spce_name",
    "specialization_url",
    "show_in_dropdown",
    "overview",
    "heading_career_options",
    "career_options",
    "heading_private_employment_area",
    "private_employment_area",
    "private_secmainhead",
    "heading_public_employment_area",
    "public_employment_area",
    "top_employers",
    "salary_trends_image",
    "heading_salary_trends",
    "salary_trends",
    "course_level",
    "list_page_content",
    "syllabus",
  ];
};
