const { dbQuery } = require("../../../config/db.config");
const tryCatch = require("../../../utils/tryCatch");
const {
  errorMsgAuth,
  errorMsgEmpty,
} = require("../../../utils/errors_message");
const { TBL_AWARDS } = require("../../../models/tables");
const updateColumn = require("../../../utils/updateColumn");
const { checkBuyerInstituteAccess } = require("../../../utils/leadAccessCheck");

const table = TBL_AWARDS;

// @desc    Get all awards
// @route   GET /api/v1/buyer/awards/:sid
// @access  Private BUYER
const getBuyerAwards = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { us_id } = req.lead;
  const { sid } = req.params;
  const { award_id, search } = req.query;
  const doesBuyerHaveAccessToSchool = await checkBuyerInstituteAccess(us_id, sid);
  if(!doesBuyerHaveAccessToSchool) errorMsgAuth("Unauthorised Access, you don't have access to this school's Awards", res)
  let qry = "";
  if (!sid) throw new Error("School id is required");
  if (search) qry += ` AND awd_title LIKE '%${search}%'`;
  let attributes = [
    `${table}.id`,
    `${table}.awd_title`,
    `${table}.awd_descriptipn`,
    `${table}.order_id`,
    `${table}.is_active`
  ];

  if (award_id) {
    attributes.push(...[`${table}.is_active`, `${table}.date`]);
    qry = ` AND id = ${award_id}`; // get one award
  }

  sql = `SELECT ${attributes.join(", ")} FROM ${table} WHERE 1=1 AND e_id = ? ${qry} ORDER BY date DESC`;
  result = await dbQuery(sql, [sid]);
  if (result.length === 0) errorMsgEmpty("No awards found", res);
  if (award_id) {
    result = result[0]; // get one award
  }
  return res.status(200).json({
    success: true,
    data: result,
  });
});

// @desc    Add award
// @route   POST /api/v1/buyer/awards/:sid
// @access  Private BUYER
const addBuyerAward = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { us_id } = req.lead;
  const { sid } = req.params;
  const { awd_title, awd_descriptipn, order_id, is_active } = req.body;
  const doesBuyerHaveAccessToSchool = await checkBuyerInstituteAccess(us_id, sid);
  if(!doesBuyerHaveAccessToSchool) errorMsgAuth("Unauthorised Access, you don't have access to this school's Awards", res)
  if (!awd_title || !awd_descriptipn || !order_id || !is_active)
    throw new Error("All fields are required");
  sql = `INSERT INTO ${table} (e_id, awd_title, awd_descriptipn, is_active, order_id, date) VALUES (?, ?, ?, ?, ?, NOW())`;
  result = await dbQuery(sql, [sid, awd_title, awd_descriptipn, is_active, order_id]);
  if (!result.affectedRows) throw new Error("Award not added");
  return res.status(200).json({
    success: true,
    message: "Award added",
  });
});

//@desc    Update award
// @route   PUT /api/v1/buyer/awards/:sid
// @access  Private BUYER
const updateBuyerAward = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { us_id } = req.lead;
  const { sid } = req.params;
  const doesBuyerHaveAccessToSchool = await checkBuyerInstituteAccess(us_id, sid);
  if(!doesBuyerHaveAccessToSchool) throw new Error("Unauthorised Access, you don't have access to this school's Awards")
  const { id, update } = req.body;
  if (!id || !update) throw new Error("All fields are required");
  const { column, value } = updateColumn(update);
  sql = `UPDATE ${table} SET ${column} WHERE id = ? AND e_id = ?`;
  result = await dbQuery(sql, [...value, id, sid]);
  if (!result.affectedRows) throw new Error("Award not updated");
  return res.status(200).json({
    success: true,
    message: "Award updated",
  });
});

//@desc    Delete award
// @route   DELETE /api/v1/buyer/awards/:sid
// @access  Private BUYER
const deleteBuyerAward = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { us_id } = req.lead;
  const { sid } = req.params;
  const doesBuyerHaveAccessToSchool = await checkBuyerInstituteAccess(us_id, sid);
  if(!doesBuyerHaveAccessToSchool) throw new Error("Unauthorised Access, you don't have access to this school's Awards")
  const { id } = req.body;
  if (!id) throw new Error("Award id is required");
  sql = `DELETE FROM ${table} WHERE id = ? AND e_id = ?`;
  result = await dbQuery(sql, [id, sid]);
  if (!result.affectedRows) throw new Error("Award not deleted");
  return res.status(200).json({
    success: true,
    message: "Award deleted",
  });
});

module.exports = {
  getBuyerAwards,
  addBuyerAward,
  updateBuyerAward,
  deleteBuyerAward,
};
