"use strict";
const { dbQuery } = require("../../../config/db.config");
const tryCatch = require("../../../utils/tryCatch");
const {
  STUDY_INDIA,
  SC_USER,
  SC_SCHOOLCITYMASTER,
} = require("../../../models/tables");
const ExcelJS = require("exceljs");
const {
  BUYER,
  GENERAL_ENQUIRY_DATA,
  ADMISSION_ENQUIRY_DATA,
} = require("../../../common/constants");
const {
  checkExportPermission,
  checkBuyerInstituteAccess,
  getBuyerInstituteInfo,
} = require("../../../utils/leadAccessCheck");

const generalTbl = STUDY_INDIA;
const admissionTbl = SC_USER;
const buyerTbl = SC_USER;
const stateCityTbl = SC_SCHOOLCITYMASTER;

// @desc Get all general/admission enquiry DATA EXPORT
// @route GET /buyer/export-enquiry/:sid/:type
// @access Private BUYER
const getStudentEnquiryDataExcel = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { us_id } = req.lead;
  const { sid } = req.params;
  const { type } = req.query;

  if (!us_id) throw new Error("Unauthorised Access");
  if (!type) throw new Error("Invalid request require type");

  const doesBuyerSchoolExist = await checkBuyerInstituteAccess(us_id, sid);
  const doesBuyerHaveExportPermission = await checkExportPermission(us_id);
  if (!doesBuyerSchoolExist || !doesBuyerHaveExportPermission)
    throw new Error(
      "Unauthorised Access, You don't have permission to access this data"
    );

  // Create a new workbook
  const workbook = new ExcelJS.Workbook();
  const worksheet = workbook.addWorksheet("Enquiry Data");

  const genrealAttributes = [];
  const admissionAttributes = [];

  switch (type) {
    case GENERAL_ENQUIRY_DATA:
      genrealAttributes.push(
        ...[
          `${generalTbl}.name`,
          `${generalTbl}.email`,
          `${generalTbl}.mobile`,
          `${generalTbl}.school_type`,
          `${generalTbl}.edu_interest`,
          `${generalTbl}.location`,
          `${generalTbl}.post_date`,
        ]
      );
      sql = `SELECT ${genrealAttributes.join(
        ", "
      )} FROM ${generalTbl} WHERE school_id = ?`;
      result = await dbQuery(sql, [sid]);
      // Worksheet Coloumns
      worksheet.columns = [
        { header: "#", key: "sr_no", width: 10 },
        { header: "Student Name", key: "name", width: 20 },
        { header: "Email", key: "email", width: 20 },
        { header: "Mobile No.", key: "mobile", width: 20 },
        { header: "School Type", key: "school_type", width: 20 },
        { header: "School Category", key: "edu_interest", width: 20 },
        { header: "Location", key: "location", width: 20 },
        { header: "Date", key: "post_date", width: 20 },
      ];
      // Add Rows to Worksheet
      result.forEach((row, index) => {
        worksheet.addRow({
          sr_no: index + 1,
          name: row.name,
          email: row.email,
          mobile: row.mobile,
          school_type: row.school_type,
          edu_interest: row.edu_interest,
          location: row.location,
          post_date: row.post_date,
        });
      });
      break;

    case ADMISSION_ENQUIRY_DATA:
      const buyerInstituteInfo = await getBuyerInstituteInfo(us_id, sid);
      const getRequiredSchoolUrl = buyerInstituteInfo.find(
        (info) => info.sid == sid
      ).school_url;
      if (!getRequiredSchoolUrl)
        throw new Error("Invalid request, No school url found");
      admissionAttributes.push(
        ...[
          `CONCAT (${admissionTbl}.us_fname, ' ', ${admissionTbl}.us_lname) as name`,
          `${admissionTbl}.email_1`,
          `${admissionTbl}.mobile_1`,
          `${admissionTbl}.parent_name AS father_name`,
          `${admissionTbl}.mother_name`,
          `${stateCityTbl}.cityName AS state`,
          // `${stateCityTbl}.cityName AS city`,
          `${admissionTbl}.us_create_date`,
          `${admissionTbl}.us_date_of_birth`,
        ]
      );
      const joinTbl = [
        `LEFT JOIN ${stateCityTbl} ON (${stateCityTbl}.cityId = ${admissionTbl}.state OR ${stateCityTbl}.cityId LIKE ${admissionTbl}.state)`,
        // `LEFT JOIN ${stateCityTbl} AS city ON (city.cityId = ${admissionTbl}.city OR city.cityId LIKE ${admissionTbl}.city)`,
      ].join(" ");
      sql = `SELECT ${admissionAttributes.join(
        ", "
      )} FROM ${admissionTbl} ${joinTbl} WHERE url = ? ;`;
      result = await dbQuery(sql, [`${getRequiredSchoolUrl}`]);
      // Worksheet Coloumns
      worksheet.columns = [
        { header: "#", key: "sr_no", width: 10 },
        { header: "Name", key: "name", width: 20 },
        { header: "Email", key: "email", width: 20 },
        { header: "Mobile No.", key: "mobile", width: 20 },
        { header: "Father Name", key: "father_name", width: 20 },
        { header: "Mother Name", key: "mother_name", width: 20 },
        { header: "State", key: "state", width: 20 },
        // { header: "City", key: "city", width: 20 },
        // { header: "DOB", key: "us_date_of_birth", width: 20 },
        { header: "Date", key: "us_create_date", width: 20 },
      ];
      // Add Rows to Worksheet
      result.forEach((row, index) => {
        worksheet.addRow({
          sr_no: index + 1,
          name: row.name,
          email: row.email_1,
          mobile: row.mobile_1,
          father_name: row.father_name,
          mother_name: row.mother_name,
          state: row.state,
          city: row.city,
          us_date_of_birth: row.us_date_of_birth,
          us_create_date: row.us_create_date,
        });
      });
      break;
    default:
      throw new Error("Invalid type");
  }

  // Generate Excel File to send to Frontend
  res.setHeader(
    "Content-Type",
    "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"
  );
  res.setHeader(
    "Content-Disposition",
    "attachment; filename=" + "Enquiry_data.xlsx"
  );
  return workbook.xlsx.write(res).then(function () {
    res.status(200).end();
  });
});

module.exports = { getStudentEnquiryDataExcel };
