const { dbQuery } = require("../../../config/db.config");
const tryCatch = require("../../../utils/tryCatch");
const {
  errorMsgAuth,
  errorMsgEmpty,
} = require("../../../utils/errors_message");
const { TBL_ACCORDION_CONTENT } = require("../../../models/tables");
const { checkBuyerInstituteAccess } = require("../../../utils/leadAccessCheck");
const { SCHOOL } = require("../../../common/constants");

const table = TBL_ACCORDION_CONTENT;

//@desc get one Faq based on id
//@route GET /api/v1/buyer/faqs/:sid/:id
//@access Private BUYER
const getBuyerFaqById = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { us_id } = req.lead;
  const { sid, id } = req.params;
  const doesBuyerHaveAccessToSchool = await checkBuyerInstituteAccess(us_id, sid);
  if (!doesBuyerHaveAccessToSchool){
    errorMsgAuth("Unauthorised Access, you don't have access to this school's FAQs", res);
  };
  const attributes = [
    `${table}.*`,
  ].join(", ");

  sql = `SELECT ${attributes} FROM ${table} WHERE 1=1 AND org_id = ? AND id = ? LIMIT 1`;
  result = await dbQuery(sql, [sid, id]);

  if (result.length === 0) errorMsgEmpty("No FAQ found", res);

  return res.status(200).json({
    success: true,
    data: result[0],
  });

});

//@desc get all Faqs of a school
//@route GET /api/v1/buyer/faqs/:sid
//@access Private BUYER
const getBuyerFaqs = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { search, page = 1, limit = 50 } = req.query;
  const offset = (page - 1) * limit;
  const { sid } = req.params;
  const { us_id } = req.lead;

  const doesBuyerHaveAccessToSchool = await checkBuyerInstituteAccess(us_id, sid);
  if (!doesBuyerHaveAccessToSchool){
    errorMsgAuth("Unauthorised Access, you don't have access to this school's FAQs", res);
  }

  let qry = "";
  if (search) qry += ` AND question LIKE '%${search}%'`;

  const attributes = [
    `${table}.id`,
    `${table}.question`,
    `${table}.answer`,
  ].join(", ");

  sql = `SELECT ${attributes} FROM ${table} WHERE 1=1 AND org_id = ? ${qry} ORDER BY date DESC LIMIT ? OFFSET ?`;
  result = await dbQuery(sql, [sid, Number(limit), offset]);

  if (result.length === 0) errorMsgEmpty("No FAQs found", res);

  const records = await dbQuery(`SELECT COUNT(*) as total FROM ${table} WHERE 1=1 AND org_id = ? ${qry}`, [sid]);

  return res.status(200).json({
    success: true,
    data: result,
    records: records[0].total,
  });

});

//@desc create a Faq
//@route POST /api/v1/buyer/faqs/:sid
//@access Private BUYER
const addBuyerFaq = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { sid } = req.params;
  const { us_id } = req.lead;
  const { question, answer, active = 'Y' } = req.body;

  if(!question || !answer) {
    res.statusCode = 400;
    throw new Error("Question and Answer are required");
  }

  const doesBuyerHaveAccessToSchool = await checkBuyerInstituteAccess(us_id, sid);
  if (!doesBuyerHaveAccessToSchool){
    errorMsgAuth("Unauthorised Access, you don't have access to this school's FAQs", res);
  };

  sql = `INSERT INTO ${table} (org_type, org_id, question, answer, active, date) VALUES (?, ?, ?, ?, ?, NOW())`;
  result = await dbQuery(sql, [SCHOOL, sid, question, answer, active]);

  return res.status(201).json({
    success: true,
    message: "FAQ created successfully",
  });

});

//@desc update a Faq based on id
//@route PUT /api/v1/buyer/faqs/:sid/:id
//@access Private BUYER
const updateBuyerFaqById = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { sid, id } = req.params;
  const { us_id } = req.lead;
  const { question, answer, active = 'Y' } = req.body;

  if(!question || !answer) {
    res.statusCode = 400;
    throw new Error("Question and Answer are required");
  };

  const doesBuyerHaveAccessToSchool = await checkBuyerInstituteAccess(us_id, sid);
  if (!doesBuyerHaveAccessToSchool){
    errorMsgAuth("Unauthorised Access, you don't have access to this school's FAQs", res);
  };

  sql = `UPDATE ${table} SET question = ?, answer = ?, active = ? WHERE id = ? AND org_id = ?`;
  result = await dbQuery(sql, [question, answer, active, id, sid]);

  return res.status(200).json({
    success: true,
    message: "FAQ updated successfully",
  });

});

//@desc delete a Faq based on id
//@route DELETE /api/v1/buyer/faqs/:sid/:id
//@access Private BUYER
const deleteBuyerFaqById = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { sid, id } = req.params;
  const { us_id } = req.lead;

  const doesBuyerHaveAccessToSchool = await checkBuyerInstituteAccess(us_id, sid);
  if (!doesBuyerHaveAccessToSchool){
    errorMsgAuth("Unauthorised Access, you don't have access to this school's FAQs", res);
  };

  sql = `DELETE FROM ${table} WHERE id = ? AND org_id = ?`;
  result = await dbQuery(sql, [id, sid]);

  return res.status(200).json({
    success: true,
    message: "FAQ deleted successfully",
  });

});

module.exports = {
  getBuyerFaqs,
  addBuyerFaq,
  getBuyerFaqById,
  updateBuyerFaqById,
  deleteBuyerFaqById,
};

// //@desc update Faq active status based on id
// //@route PUT /api/v1/buyer/faqs/:sid/:id/active
// //@access Private BUYER
// const updateBuyerFaqActiveStatusById = tryCatch(async (req, res) => {
//   let [sql, result] = [null, null];
//   const { sid, id } = req.params;
//   const { us_id } = req.lead;
//   const { active } = req.body;

//   const doesBuyerHaveAccessToSchool = await checkBuyerInstituteAccess(us_id, sid);
//   if (!doesBuyerHaveAccessToSchool){
//     errorMsgAuth("Unauthorised Access, you don't have access to this school's FAQs", res);
//   };

//   sql = `UPDATE ${table} SET active = ? WHERE id = ? AND org_id = ?`;
//   result = await dbQuery(sql, [active, id, sid]);

//   return res.status(200).json({
//     success: true,
//     message: "FAQ active status updated successfully",
//   });

// });