const { dbQuery } = require("../../../config/db.config");
const tryCatch = require("../../../utils/tryCatch");
const { errorMsgAuth } = require("../../../utils/errors_message");
const { SC_USER, STUDY_INDIA } = require("../../../models/tables");
const { getBuyerInstituteInfo, checkExportPermission } = require("../../../utils/leadAccessCheck");

const table = SC_USER;
const leadTbl = STUDY_INDIA;
const admissionTbl = SC_USER; // We will take those user who have applied for admission in this institute type school/college

const getLeadsFromGeneralEnquiry = async (schoolIds) => {
  try {
    if (!schoolIds.length) return 0;
    let [sql, result] = [null, null];
    sql = `SELECT COUNT(*) AS leads FROM ${leadTbl} WHERE school_id IN (?)`;
    result = await dbQuery(sql, [...schoolIds]);
    return result[0].leads;
  } catch (error) {
    throw new Error(error);
  }
};

const getLeadAdmissionFromUsers = async (schoolUrls) => {
  try {
    if (!schoolUrls.length) return 0;
    let [sql, result] = [null, null];
    sql = `SELECT COUNT(*) AS admissions FROM ${admissionTbl} WHERE url IN (?)`;
    result = await dbQuery(sql, [...schoolUrls]);
    return result[0].admissions;
  } catch (error) {
    throw new Error(error);
  }
};

const getBuyersSubscriptionDetails = async (id) => {
  try {
    let [sql, result] = [null, null];
    sql = `SELECT subscription_date, renewal_date FROM ${table} WHERE us_id = ? LIMIT 1`;
    result = await dbQuery(sql, [id]);
    const { subscription_date, renewal_date } = result[0];
    return { subscription_date, renewal_date };
  } catch (error) {
    throw new Error(error);
  }
};

// @desc    Get Dashboard details of buyer
// @route   GET /buyer/home
// @access  Private Buyer
const getDashboardDetails = tryCatch(async (req, res) => {


  
  const  us_id  = req.query.us_id;

  
  if (!us_id) errorMsgAuth("Unauthorised Access", res);

  const data = {};
  

  const buyersInstituteDetails = await getBuyerInstituteInfo(us_id);

  const buyersSchoolIds = buyersInstituteDetails.map((school) => school.sid);
  const buyersSchoolUrls = buyersInstituteDetails.map((school) => school.school_url);

  // Get Main School Name and logo
  data["main_school"] = buyersInstituteDetails[0]?.school_name || "";
  data["main_school_logo"] = buyersInstituteDetails[0]?.school_logo || ""; 

  // Get Total Leads from general enquiry
  const leads = await getLeadsFromGeneralEnquiry(buyersSchoolIds);
  data["leads"] = leads;

  // Get Total Admissions from users
  const admissions = await getLeadAdmissionFromUsers(buyersSchoolUrls);
  data["admissions"] = admissions;
  
  // Check Export Permission
  const exportPermission = await checkExportPermission(us_id);
  data["export_permission"] = exportPermission === true ? "Y" : "N";

  // Get Subscription Details
  const subscriptionDetails = await getBuyersSubscriptionDetails(us_id);
  data["subscription_details"] = subscriptionDetails;


  res.status(200).json({
    success: true,
    data,
  });
});

module.exports = { getDashboardDetails };
