const { dbQuery } = require("../../../config/db.config");
const tryCatch = require("../../../utils/tryCatch");
const { errorMsgAuth, errorMsgEmpty } = require("../../../utils/errors_message");
const updateColumn = require("../../../utils/updateColumn");
const { SCHOOL_DETAIL, TBL_SCHOOL_GRADE_FEE } = require("../../../models/tables");
const { urlExist } = require("../../../utils/url_exist");
const { getGradesFeesSortObj } = require("../../admin/CMS/SchoolDetails/s_grade_fees.controller");
const { checkBuyerInstituteAccess, getBuyerInstituteInfo, checkSchoolGradeFeesExist } = require("../../../utils/leadAccessCheck");

const table = SCHOOL_DETAIL;
const schoolGeadeTbl = TBL_SCHOOL_GRADE_FEE

// @desc    Get all institute id and details of buyer
// @route   GET /buyer/institute
// @access  Private Buyer
const getBuyerInstitutes = tryCatch(async (req, res) => {
  // const { us_id } = req.lead;

  const us_id=req.query.us_id
  if (!us_id) errorMsgAuth("Unauthorised Access", res);

  const instituteDetails = await getBuyerInstituteInfo(us_id);
  if (instituteDetails.length === 0)errorMsgAuth("No institute found, contact allschoolscolleges.com", res);

  return res.status(200).json({
    success: true,
    data: instituteDetails,
  });
});

// @desc    Get institute details of buyer
// @route   GET /buyer/institute-details/:sid
// @access  Private Buyer
const getBuyerInstituteDetails = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];

  
  // const { us_id, user_type } = req.lead;

  const us_id=req.query.us_id
  const user_type='school'

  

  

  if (!us_id || !user_type) errorMsgAuth("Unauthorised Access", res);
  const { sid } =  req.params;

  const doesBuyerHaveAccessToSchool = await checkBuyerInstituteAccess(us_id, sid);
  if(!doesBuyerHaveAccessToSchool) errorMsgAuth("Unauthorised Access, You don't have permission for this school details", res);

  const attributes = [`${table}.*`];
  sql = `SELECT ${attributes.join( ", ")} FROM ${table} WHERE buyer_name = ? AND sid = ? LIMIT 1`;
  result = await dbQuery(sql, [us_id, sid]);
  if (result.length === 0)
    errorMsgAuth("No institute details found, contact allschoolscolleges.com", res);
  return res.status(200).json({
    success: true,
    data: result[0],
  });
});

// @desc    Update institute details of buyer
// @route   PUT /buyer/institute-details/:sid
// @access  Private Buyer
const updateBuyerInstituteDetails = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { us_id } = req.lead;
  if (!us_id) errorMsgAuth("Unauthorised Access", res);
  const { sid } = req.params;
  const doesBuyerHaveAccessToSchool = await checkBuyerInstituteAccess(us_id, sid);
  if(!doesBuyerHaveAccessToSchool) errorMsgAuth("Unauthorised Access, You don't have permission for this school details", res);
  const { update } = req.body;
  if (!update) throw new Error("Update is required");
  update?.school_url && (await urlExist(table, "sid", "school_url", update.school_url, sid));
  delete update?.sid;
  delete update?.buyer_name;
  const { column, value } = updateColumn(update);
  sql = `UPDATE ${table} SET ${column} WHERE buyer_name = ? AND sid = ?`;
  result = await dbQuery(sql, [...value, us_id, sid]);

  if (!result.affectedRows)
    throw new Error( "No institute details updated, contact allschoolscolleges.com");

  return res.status(200).json({
    success: true,
    message: "Institute details updated successfully",
  });
});

// @desc    Get institute fees details of buyer
// @route   GET /buyer/grade-fee/:sid
// @access  Private Buyer
const getBuyerSchoolFeesDetails = tryCatch(async (req, res) => {
  const { us_id } = req.lead;
  if (!us_id) throw new Error("Unauthorised Access");
  const { sid } = req.params;
  const doesBuyerHaveAccessToSchool = await checkBuyerInstituteAccess(us_id, sid);
  if(!doesBuyerHaveAccessToSchool) throw new Error("School doesn't exist or you don't have permission for this school details")
  const doesSchoolGradesFeesExists = await checkSchoolGradeFeesExist(sid);
  if(!doesSchoolGradesFeesExists) errorMsgEmpty("School grade fees not found",res);
  const attributes = [
    `${schoolGeadeTbl}.id`,
    `${schoolGeadeTbl}.grade`,
    `${schoolGeadeTbl}.fee`,
  ]
  sql = `SELECT ${attributes.join(", ")}  FROM ${schoolGeadeTbl} WHERE sid = ? ORDER BY ${schoolGeadeTbl}.sort ASC`;
  result = await dbQuery(sql, [sid]);

  return res.status(200).json({
    success: true,
    data: result,
  });
});

// @desc    Add institute fees details of buyer
// @route   POST /buyer/grade-fee/:sid
// @access  Private Buyer
const addBuyerSchoolFeesDetails = tryCatch(async (req, res) => {
  const { us_id } = req.lead;
  if (!us_id) throw new Error("Unauthorised Access");
  const { sid } = req.params;

  const doesBuyerHaveAccessToSchool = await checkBuyerInstituteAccess(us_id, sid);
  if(!doesBuyerHaveAccessToSchool) throw new Error("School doesn't exist or you don't have permission for this school details")
  const { grade_fee } = req.body;
  if(!grade_fee) throw new Error("Grade and fee is required");

  const doesSchoolGradesFeesExists = await checkSchoolGradeFeesExist(sid);
  if(doesSchoolGradesFeesExists) {
   const deleteExistingGradesFeesSql = `DELETE FROM ${schoolGeadeTbl} WHERE sid = ?`;
   const deleteExistingGradesFeesResult = await dbQuery(deleteExistingGradesFeesSql, [sid]);
   if(!deleteExistingGradesFeesResult.affectedRows) throw new Error("Something went wrong, while deleting existing school grade fees");
  }
  
  const gradesFeesSortObj = getGradesFeesSortObj(grade_fee);
  const attributes = [
    `${schoolGeadeTbl}.sid`,
    `${schoolGeadeTbl}.grade`,
    `${schoolGeadeTbl}.fee`,
    `${schoolGeadeTbl}.sort`,
  ];
  const values = Object.values(gradesFeesSortObj).map((obj) => {
    return [sid, obj.grade, obj.fee, obj.sort];
  });
  sql = `INSERT INTO ${schoolGeadeTbl} (${attributes.join(", ")}) VALUES ?`;
  result = await dbQuery(sql, [values]);

  if (!result.affectedRows) throw new Error("Something went wrong while adding school grade fees");

  return res.status(200).json({
    success: true,
    message: "School grade fees added successfully",
  });
});

module.exports = {
  getBuyerInstitutes,
  getBuyerInstituteDetails,
  updateBuyerInstituteDetails,
  getBuyerSchoolFeesDetails,
  addBuyerSchoolFeesDetails,
};
