const {
  getArticlesService,
  getArticlesCategoryService,
} = require("../../../services/article.service");
const {
  SC_ASC_ARTICLES,
  SC_ARTICLES_CATEGORY,
} = require("../../../models/tables");
const tryCatch = require("../../../utils/tryCatch");

// @desc get articles category for public display
// @route GET /api/v1/public/articles-category
// @access PUBLIC
const getArticlesCategory = tryCatch(async (req, res) => {
  const queryMap = {};
  queryMap.attributes = [
    `${SC_ARTICLES_CATEGORY}.nc_id AS value`,
    `${SC_ARTICLES_CATEGORY}.nc_name AS name`,
    `${SC_ARTICLES_CATEGORY}.nc_mtitle AS meta_title`,
    `${SC_ARTICLES_CATEGORY}.nc_mkeyword AS meta_keyword`,
    `${SC_ARTICLES_CATEGORY}.nc_mdescription AS meta_description`,
    `${SC_ARTICLES_CATEGORY}.nc_front_url AS url`
  ];
  queryMap.conditions = [`${SC_ARTICLES_CATEGORY}.nc_active = 'Y'`];
  queryMap.additional = [`ORDER BY ${SC_ARTICLES_CATEGORY}.nc_priority ASC`];

  const result = await getArticlesCategoryService(queryMap);
  if (result instanceof Error || result.length === 0) {
    res.statusCode = 400;
    throw new result;
  }

  return res.status(200).json({
    success: true,
    data: result,
  });
});

// @desc get one article for public display based on the url
// @route GET /api/v1/public/article/:url
// @access PUBLIC
const getOneArticleDetails = tryCatch(async (req, res) => {
  const { url } = req.params;

  const queryMap = {};
  queryMap.attributes = [
    `${SC_ASC_ARTICLES}.nr_id`,
    `${SC_ASC_ARTICLES}.nr_title`,
    `${SC_ASC_ARTICLES}.nr_front_url`,
    `${SC_ASC_ARTICLES}.nr_image`,
    `${SC_ASC_ARTICLES}.nr_large_description`,
    `${SC_ASC_ARTICLES}.nr_mtitle`,
    `${SC_ASC_ARTICLES}.nr_mdescription`,
    `${SC_ASC_ARTICLES}.nr_mkeyword`,
  ];
  queryMap.conditions = [
    `${SC_ASC_ARTICLES}.nr_active = 'Y'`,
    `${SC_ASC_ARTICLES}.nr_front_url = '${url}'`,
  ];
  queryMap.additional = [`LIMIT 1`];

  const result = await getArticlesService(queryMap);
  if (result instanceof Error || result.length === 0) {
    res.statusCode = 404;
    throw new Error("No data found");
  }

  return res.status(200).json({
    success: true,
    data: result[0],
  });
});

// @desc get all the articles for public display
// @route GET /api/v1/public/articles?url
// @access PUBLIC
const getArticlesPage = tryCatch(async (req, res) => {
  const { url } = req.query;

  const queryMap = {};
  queryMap.attributes = [
    `${SC_ASC_ARTICLES}.nr_id`,
    `${SC_ASC_ARTICLES}.nr_title`,
    `${SC_ASC_ARTICLES}.nr_front_url`,
    `${SC_ASC_ARTICLES}.nr_image`,
    `${SC_ASC_ARTICLES}.nr_description`,
    `${SC_ASC_ARTICLES}.nr_publish_date`,
  ];
  queryMap.conditions = [`${SC_ASC_ARTICLES}.nr_active = 'Y'`];
  queryMap.additional = [`ORDER BY ${SC_ASC_ARTICLES}.nr_id desc`];

    // Get category id from url
    if(url){
      const categoryIdFromUrl = await getArticlesCategoryService({
        attributes: [`${SC_ARTICLES_CATEGORY}.nc_id`],
        conditions: [`${SC_ARTICLES_CATEGORY}.nc_front_url = ?`, `${SC_ARTICLES_CATEGORY}.nc_active = 'Y'`],
        parameters:[url],
        additional: [`LIMIT 1`],
      })
      const categoryFromUrl = categoryIdFromUrl.pop();
      if(!categoryFromUrl){
        res.statusCode = 404;
        throw new Error("Category not found")
      }else{
        queryMap.conditions.push(`${SC_ASC_ARTICLES}.nc_id = ${categoryFromUrl.nc_id}`)
      }
    }

  const result = await getArticlesService(queryMap);
  if (result instanceof Error || result.length === 0) {
    res.statusCode = 404;
    throw new Error("No data found");
  }

  return res.status(200).json({
    success: true,
    data: result,
  });
});

// @desc get related articles top 6
// @route GET /api/v1/public/articles-related
// @access PUBLIC
const getRelatedArticles = tryCatch(async (req, res) => {
  const queryMap = {};
  queryMap.attributes = [
    `${SC_ASC_ARTICLES}.nr_id`,
    `${SC_ASC_ARTICLES}.nr_title`,
    `${SC_ASC_ARTICLES}.nr_front_url`,
    `${SC_ASC_ARTICLES}.nr_image`,
  ];
  queryMap.conditions = [`${SC_ASC_ARTICLES}.nr_active = 'Y'`];
  queryMap.additional = [`ORDER BY ${SC_ASC_ARTICLES}.nr_priority ASC LIMIT 6`];

  const result = await getArticlesService(queryMap);
  if (result instanceof Error || result.length === 0) {
    res.statusCode = 404;
    throw new Error("No data found");
  }

  return res.status(200).json({
    success: true,
    data: result,
  });
});

module.exports = {
  getArticlesPage,
  getOneArticleDetails,
  getRelatedArticles,
  getArticlesCategory,
};
