const { dbQuery } = require("../../../config/db.config");
const {
  C_COURSE,
  C_INSTITUTE,
  C_CONTACT,
  C_CONTENT,
  STATE_TBL,
  CITY_TBL,
  INSTITUTE_TYPE,
  COURSELEVEL,
  COURSE_DELIVERY,
  SPECIALIZATION,
  COURSE,
} = require("../../../models/tables");
const { sanitizedUrlFromParams } = require("../../../utils/sanitizedUrl");

const {
  getCollegeService,
  getCollegesContentService,
  getCollegesCourseService,
} = require("../../../services/college.service");
const tryCatch = require("../../../utils/tryCatch");

// @desc get details on multiple college based on url category, its for public to view
// @route GET /api/v1/public/colleges/?url=college-url
// @access PUBLIC
const getCollegesListingForPublic = tryCatch(async (req, res) => {
  // console.log(req.params, req.query)

   

 

  const { url, course } = req.params;
  const { limit = 0, offset = 20 } = req.query;

  


  if (!url) {
    return res.status(404).json({
      status: "fail",
      message: "Listing not found",
      data: [],
    });
  }

  // Get Content for those required colleges list based on URL
  const queryMapCollegeContent = {
    attributes: [`${C_CONTENT}.brief_desc`, `${C_CONTENT}.full_desc`],
    conditions: [`${C_CONTENT}.on_url LIKE ?`, `${C_CONTENT}.is_active = "Y"`],
    additional: [`LIMIT 1`],
    parameters: [`%/${url}`],
  };

  const contentData = await getCollegesContentService(queryMapCollegeContent);
  let { location, category } = sanitizedUrlFromParams(course);

  


   
  
  // Get colleges list based on URL category
  // Join college details, course details, location details, college content
  const queryMapCollege = {
    attributes: [
      `${C_INSTITUTE}.e_id`,
      `${C_INSTITUTE}.institute_name AS name`,
      `${C_INSTITUTE}.establish`,
      `${C_INSTITUTE}.institute_logo AS logo`,
      `${C_INSTITUTE}.institute_mainimg AS mainImg`,
      `${C_INSTITUTE}.is_partner`,
      `${C_INSTITUTE}.institute_url AS url`,
      `${C_INSTITUTE}.priority`,
      `${C_CONTACT}.state`,
      `${C_CONTACT}.city`,
    ],
    conditions: [
      `${C_INSTITUTE}.active = 'Y'`,
      `(${C_COURSE}.course_category LIKE ? AND( ${C_COURSE}.course_name LIKE ? OR ${C_COURSE}.specialization LIKE ? ))`,
    ],
    parameters: [
      ...[
        //`%${url}%`,
        `%${url.replace(/-/g, ' ')}%`,
        `%${category.replace(/-/g, ' ')}%`,
        `%${category.replace(/-/g, ' ')}%`,
      ],
    ],
    additional: [
      `GROUP BY ${C_INSTITUTE}.e_id`,
      `ORDER BY ${C_INSTITUTE}.is_partner DESC, ${C_INSTITUTE}.priority ASC`,
      `LIMIT 0, ${Number(offset)}`,
    ],
    joinTbl: [
      `INNER JOIN ${C_COURSE} ON ${C_INSTITUTE}.e_id = ${C_COURSE}.e_id`,
      `INNER JOIN ${C_CONTACT} ON ${C_CONTACT}.e_id = ${C_COURSE}.e_id`,
    ],
  };

  if (location && location !== "india") {
    queryMapCollege.conditions.push(
      `(${C_CONTACT}.state LIKE ? OR ${C_CONTACT}.city LIKE ?)`
    );
    queryMapCollege.parameters.push(`%${location}%`, `%${location}%`);
  }

  pushRequiredFilters(req.query, queryMapCollege);

  const collegeResult = await getCollegeService(queryMapCollege);
  const collegesResultWithTopCourse = await setCollegesWithTopCourse(
    collegeResult,
    category
  );

  const sendData = {};
  sendData.page_description = contentData[0] || null;
  sendData.colleges = collegesResultWithTopCourse;

  
  return res.status(200).json({
    status: "success",
    data: sendData,
  });
});

// @desc search for college name and get name and url for public to view limited to 10
// @route GET /api/v1/public/find/colleges?search=college-name
// @access PUBLIC
const getCollegesSearchForPublic = tryCatch(async (req, res) => {
  const { search } = req.query;
// console.log("req.query",req.query)
  
  if (!search) {
    return res.status(400).json({
      status: "fail",
      message: "Please provide search query",
    });
  }
  const queryMap = {
    attributes: [
      `${C_INSTITUTE}.institute_name AS name`,
      `${C_INSTITUTE}.institute_url AS url`,
    ],
    conditions: [`${C_INSTITUTE}.institute_name LIKE ?`],
    parameters: [`%${search}%`],
    additional: [`LIMIT 10`],
  };

  const result = await getCollegeService(queryMap);

  
  return res.status(200).json({
    status: "success",
    data: result,
  });
});

module.exports = {
  getCollegesListingForPublic,
  getCollegesSearchForPublic,
};

const pushRequiredFilters = (query, { conditions, parameters, joinTbl }) => {
  const courseCategory = query.course_category;
  if (courseCategory) {
    conditions.push(`${C_COURSE}.course_category LIKE ?`);
    parameters.push(`%${courseCategory}%`);
  }

  const courseName = query.course_name;
  if (courseName) {
    conditions.push(
      `(${COURSE}.course_id = ? OR ${COURSE}.course_name LIKE ?)`
    );
    parameters.push(`${courseName}`, `%${courseName}%`);
    joinTbl.push(
      `LEFT JOIN ${COURSE} ON (${COURSE}.course_id = ${C_COURSE}.course_name OR ${COURSE}.course_name = ${C_COURSE}.course_name)`
    );
  }

  const spceName = query.spce_name;
  if (spceName) {
    conditions.push(
      `(${SPECIALIZATION}.spec_id = ? OR ${SPECIALIZATION}.spce_name LIKE ?)`
    );
    parameters.push(`${spceName}`, `%${spceName}%`);
    joinTbl.push(
      `LEFT JOIN ${SPECIALIZATION} ON (${SPECIALIZATION}.spec_id = ${C_COURSE}.specialization OR ${SPECIALIZATION}.spce_name = ${C_COURSE}.specialization)`
    );
  }

  const state = query.state;
  if (state) {
    conditions.push(`(${STATE_TBL}.sid = ? OR ${STATE_TBL}.state LIKE ?)`);
    parameters.push(`${state}`, `%${state}%`);
    joinTbl.push(
      `LEFT JOIN ${STATE_TBL} ON (${STATE_TBL}.sid = ${C_CONTACT}.state OR ${STATE_TBL}.state = ${C_CONTACT}.state)`
    );
  }

  const city = query.city;
  if (city) {
    conditions.push(`(${CITY_TBL}.ctid = ? OR ${CITY_TBL}.cname LIKE ?)`);
    parameters.push(`${city}`, `%${city}%`);
    joinTbl.push(
      `LEFT JOIN ${CITY_TBL} ON (${CITY_TBL}.ctid = ${C_CONTACT}.city OR ${CITY_TBL}.cname = ${C_CONTACT}.city)`
    );
  }

  const courseDelivery = query.course_delivery;
  if (courseDelivery) {
    conditions.push(
      `(${COURSE_DELIVERY}.id = ? OR ${COURSE_DELIVERY}.name LIKE ?)`
    );
    parameters.push(`${courseDelivery}`, `%${courseDelivery}%`);
    joinTbl.push(
      `LEFT JOIN ${COURSE_DELIVERY} ON (${COURSE_DELIVERY}.id = ${C_COURSE}.mode_learning OR ${COURSE_DELIVERY}.name = ${C_COURSE}.mode_learning)`
    );
  }

  const collegeType = query.college_type;
  if (collegeType) {
    conditions.push(
      `(${INSTITUTE_TYPE}.id = ? OR ${INSTITUTE_TYPE}.name LIKE ?)`
    );
    parameters.push(`${collegeType}`, `%${collegeType}%`);
    joinTbl.push(
      `LEFT JOIN ${INSTITUTE_TYPE} ON (${INSTITUTE_TYPE}.id = ${C_INSTITUTE}.college_type OR ${INSTITUTE_TYPE}.name = ${C_INSTITUTE}.college_type)`
    );
  }

  const courseLevel = query.course_level;
  if (courseLevel) {
    conditions.push(`(${COURSELEVEL}.id = ? OR ${COURSELEVEL}.name LIKE ?)`);
    parameters.push(`${courseLevel}`, `%${courseLevel}%`);
    joinTbl.push(
      `LEFT JOIN ${COURSELEVEL} ON (${COURSELEVEL}.id = ${C_COURSE}.course_level OR ${COURSELEVEL}.name = ${C_COURSE}.course_level)`
    );
  }

  return null;
};

const setCollegesWithTopCourse = async (colleges, category) => {
  const collegesWithTopCourse = [];
  for (let i = 0; i < colleges.length; i++) {
    const college = colleges[i];
    const { e_id } = college;
    const queryMapCollegeCourse = {
      attributes: [
        `${C_COURSE}.course_name`,
        `${C_COURSE}.course_duration`,
        `${C_COURSE}.course_duration_type`,
        `${C_COURSE}.course_fee`,
        `${C_COURSE}.specialization`,
      ],
      conditions: [
        `${C_COURSE}.e_id = ?`,
        `(${C_COURSE}.course_category LIKE ? OR ${C_COURSE}.course_name LIKE ? OR ${C_COURSE}.course_name LIKE ?)`,
      ],
      parameters: [
        e_id,
        ...[
          `%${category}%`,
          `%${category}%`,
          `%${category.substring(0, 1) + "." + category.substring(1)}%`,
        ],
      ],
      additional: [`LIMIT 5`],
    };
    const courseResult = await getCollegesCourseService(queryMapCollegeCourse);
    collegesWithTopCourse.push({
      ...college,
      courses: courseResult,
    });
  }
  return collegesWithTopCourse;
};
