const { TBL_EXAM_COURSE, CATEGORY } = require("../../../models/tables");
const {
  getExamCategoriesService,
  getExamsByCategoryIdService,
} = require("../../../services/exam.service");
const tryCatch = require("../../../utils/tryCatch");

const examTbl = TBL_EXAM_COURSE;
const categoryTbl = CATEGORY;

// @desc GET all exams for listing public
// @route GET /api/v1/public/exams
// @access PUBLIC
const getExamsForListing = tryCatch(async (req, res) => {
  const categories = await getExamCategories();
  // Sanitize categories into Objects
  const categoriesObj = {};
  categories.forEach((category) => {
    categoriesObj[category.cat_id] = category;
  });

  // Multiple exams will be fetched for each category id
  const examsPromises = categories.map((category) =>
    getExamsByCategoryId(category.cat_id)
  );
  const examsResults = await Promise.all(examsPromises);
  const examsObj = {};
  examsResults.forEach((exams, index) => {
    examsObj[categories[index].cat_id] = exams;
  });

  const mainExamListing = [];

  Object.keys(examsObj).forEach((key) => {
    mainExamListing.push({
      exam_category: categoriesObj[key].cat_name,
      exam_title: categoriesObj[key].cat_name + " Entrance Exams",
      exam_img: categoriesObj[key].cat_image,
      exams_available: examsObj[key],
    });
  });

  return res.status(200).json({
    status: "success",
    message: "Exams fetched successfully",
    data: {
      exams: mainExamListing,
    },
  });
});

// @desc search for exams by name and get its name and url LIMIT 10
// @route GET /api/v1/public/find/exams?search=exam_name
// @access PUBLIC
const getExamsSearch = tryCatch(async (req, res) => {
  
  const { search } = req.query;
  const queryMap = {};
  queryMap.attributes = [`${examTbl}.exam_name AS name`, `${examTbl}.exam_url AS url`];
  queryMap.conditions = [
    `${examTbl}.exam_name LIKE ?`,
    `${examTbl}.active = 'Y'`,
  ];
  queryMap.additional = [`LIMIT 10`];
  queryMap.parameters = [`${search}%`];

  const result = await getExamsByCategoryIdService(queryMap);
  if (result instanceof Error) {
    throw new Error(result);
  }

  return res.status(200).json({
    status: "success",
    message: "Exams fetched successfully",
    data: result,
  });
});

module.exports = { getExamsForListing, getExamsSearch };

// getting the exam category from category table
const getExamCategories = async () => {
  const queryMap = {};
  queryMap.attributes = [
    `${categoryTbl}.cat_name`,
    `${categoryTbl}.cat_id`,
    `${categoryTbl}.cat_url`,
    `${categoryTbl}.cat_image`,
  ];
  queryMap.conditions = [
    `${categoryTbl}.active = 'Y'`,
    `${categoryTbl}.isExam = 'Y'`,
  ];
  queryMap.additional = [`ORDER BY ${categoryTbl}.priority ASC`];

  try {
    const result = await getExamCategoriesService(queryMap);
    if (result instanceof Error) {
      throw new Error(result);
    }

    return result;
  } catch (error) {
    throw new Error(error);
  }
};

// getting exams from exam table using category
const getExamsByCategoryId = async (catId) => {
  const queryMap = {};
  queryMap.attributes = [`${examTbl}.exam_name`, `${examTbl}.exam_url`];
  queryMap.conditions = [
    `FIND_IN_SET(${catId}, category)`,
    ` ${examTbl}.active = 'Y'`,
  ];

  try {
    const result = await getExamsByCategoryIdService(queryMap);
    return result;
  } catch (error) {
    throw new Error(error);
  }
};
