const { dbQuery } = require("../../../../config/db.config");
const tryCatch = require("../../../../utils/tryCatch");
const updateColumn = require("../../../../utils/updateColumn");
const {
  errorMsgEmpty,
  errorMsgAuth,
} = require("../../../../utils/errors_message");
const { SC_BANK_DETAIL } = require("../../../../models/tables");
const { filterRequiredFields, exportFieldsName } = require("../../requiredFieldsForUpdate.util");
 
const table = SC_BANK_DETAIL;

// @desc Get student bank detail
// @route GET /api/v1/student/bank-detail based on us_id
// @access Private STUDENT
const get_studentBankDetail = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { us_id } = req.lead;
  if (!us_id) errorMsgAuth("Unauthorized access", res);
  const attributes = [
    `${table}.bank_ac_holder`,
    `${table}.bank_name`,
    `${table}.bank_ac`,
    `${table}.bank_ifsc`,
    `${table}.upi`
  ]
  sql = `SELECT ${attributes.join(", ")} FROM ${table} WHERE us_id = ? LIMIT 1`;
  result = await dbQuery(sql, [us_id]);
  if (result.length === 0) {
    return res.status(200).json({ success: true, data: null });
  }
  return res.status(200).json({ success: true, data: result[0] });
});

const post_studentBankDetail = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { us_id } = req.lead;
  const {
    bank_ac_holder,
    bank_name,
    bank_ac,
    bank_ifsc,
    upi = null,
  } = req.body;
  if (!us_id) errorMsgAuth("Unauthorized access", res);
  if (!bank_ac_holder || !bank_name || !bank_ac || !bank_ifsc)
    throw new Error("Please fill all the fields");
  const attributes = [
    `${table}.us_id`,
    `${table}.bank_ac_holder`,
    `${table}.bank_name`,
    `${table}.bank_ac`,
    `${table}.bank_ifsc`,
    `${table}.upi`
  ]
  const insertBody = [
    us_id,
    bank_ac_holder,
    bank_name,
    bank_ac,
    bank_ifsc,
    upi,
  ]
  sql = `INSERT INTO ${table} (${attributes.join(", ")}) VALUES (?, ?, ?, ?, ?, ?)`;
  result = await dbQuery(sql, insertBody);
  if (result.affectedRows === 0) throw new Error("Unable to add bank detail");
  return res.status(200).json({ success: true, message: "Bank detail added" });
});

const update_studentBankDetail = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { us_id } = req.lead;
  const { update } = req.body;
  if (!us_id || !update) errorMsgAuth("Unauthorized access", res);
  const requiredFields = filterRequiredFields(update, exportFieldsName.fieldsForCollegeBankDetails);
  const { column, value } = updateColumn(requiredFields);
  sql = `UPDATE ${table} SET ${column} WHERE us_id = ?`;
  result = await dbQuery(sql, [...value, us_id]);
  if (result.affectedRows === 0)
    throw new Error("Unable to update bank detail");
  return res
    .status(200)
    .json({ success: true, message: "Bank detail updated" });
});

const delete_studentBankDetail = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { us_id } = req.lead;
  if (!us_id) errorMsgAuth("Unauthorized access", res);
  const { confirm = "no" } = req.body;
  if (confirm !== "yes") throw new Error("Please confirm to delete");
  sql = `DELETE FROM ${table} WHERE us_id = ?`;
  result = await dbQuery(sql, [us_id]);
  if (result.affectedRows === 0)
    throw new Error("Unable to delete bank detail");
  return res
    .status(200)
    .json({ success: true, message: "Bank detail deleted" });
});

module.exports = {
  get_studentBankDetail,
  post_studentBankDetail,
  update_studentBankDetail,
  delete_studentBankDetail,
};
