const { dbQuery } = require("../../../../config/db.config");
const tryCatch = require("../../../../utils/tryCatch");
const {
  C_COURSE,
  TBL_PARTNER_STUDENT_QUERY,
  SC_USER,
  PARTNER_STUDENT_DOC,
  C_INSTITUTE,
  COURSE,
} = require("../../../../models/tables");
const { getEnquiryService } = require("../../../../services/enquiry.service");
const { getCollegeService, getCollegesCourseService } = require("../../../../services/college.service");
const { getScUserService } = require("../../../../services/sc_user.service");
const { getCourseService } = require("../../../../services/master/course.service")


// @desc Get Student College Dashboard details
// @route GET /api/v1/student/dashboard
// @access Private STUDENT
const getClgDashboardDetails = tryCatch(async (req, res) => {

  const { us_id, user_type } = req.lead;
  if (!us_id || !user_type){
    res.statusCode = 400;
    throw new Error("Unauthorized access");
  };

  const totalQueryRaised = await getQueryRaisedByUser(us_id);

  const { userFormStatus, courseApplied, collegeUrl } = await getApplicationStatusOfUser(us_id);

  const data = {
    status: {
      totalQueryRaised,
      userFormStatus,
      application_in_progress: 1, // TODO: Need to check later on
    },
    application_course_details: await getLeadCourseDetails(collegeUrl, courseApplied),
    recommended_colleges: await getRecommendedCollegesForUser(),
  };

  return res.status(200).json({
    success: true,
    data,
  });

});

module.exports = {
  getClgDashboardDetails,
};

// Helper functions

const getRecommendedCollegesForUser = async () => {
  const queryMap = {
    attributes: [
      `${C_INSTITUTE}.e_id`,
      `${C_INSTITUTE}.institute_name AS name`,
      `${C_INSTITUTE}.institute_logo AS logo`,
      `${C_INSTITUTE}.institute_short_name`,
      `${C_INSTITUTE}.institute_url AS url`,
    ],
    conditions: [
      `${C_INSTITUTE}.is_partner = '1'`,
      `${C_INSTITUTE}.active = 'Y'`,
    ],
    additional: [
      `ORDER BY ${C_INSTITUTE}.priority ASC`,
      `LIMIT 3`,
    ],
  };

  const result = await getCollegeService(queryMap);
  if(result instanceof Error || !result[0]){
    throw new Error("ERROR IN GETTING RECOMMENDED COLLEGES");
  }

  return result;

};

const getQueryRaisedByUser = async (id) => {
  try {
    const queryMap = {
      attributes: [`COUNT(*) AS total`],
      conditions: [`us_id = ?`],
      parameters: [id],
      table: TBL_PARTNER_STUDENT_QUERY,
    }
    const result = await getEnquiryService(queryMap);
    if(!result){
      return  0;
    }
    return result[0].total;
  } catch (error) {
    throw new Error(error)
  }
}; 

const getApplicationStatusOfUser = async (id) => {

  const queryMap = {
    attributes: [
      `${SC_USER}.us_fname`,
      `${SC_USER}.us_username`,
      `${SC_USER}.email_1`,
      `${SC_USER}.mobile_1`,
      `${SC_USER}.us_date_of_birth`,
      `${SC_USER}.us_gender`,
      `${SC_USER}.course`,
      `${SC_USER}.student_aadhar_no`,
      `${PARTNER_STUDENT_DOC}.profile_pic`,
      `${PARTNER_STUDENT_DOC}.aadhar_front`,
      `${PARTNER_STUDENT_DOC}.aadhar_back`,
      `${PARTNER_STUDENT_DOC}.ten_marksheet`,
      `${SC_USER}.course`,
      `${SC_USER}.url AS url`,
    ],
    conditions: [`${SC_USER}.us_id = ?`],
    parameters: [id],
    additional: [`LIMIT 1`],
    joinTbl: [`LEFT JOIN ${PARTNER_STUDENT_DOC} ON ${SC_USER}.us_id = ${PARTNER_STUDENT_DOC}.us_id`],
  };

  const result = await getScUserService(queryMap);
  if(result instanceof Error || !result[0]){
    res.statusCode = 400;
    throw new Error("ERROR IN GETTING APPLICATION STATUS");
  }

  const userForm = result.pop();
  const collegeUrl = userForm.url;
  const courseApplied = userForm.course;

  delete userForm.url;
  const userFormStatus = Object.values(userForm).every((val) => val !== null);

  return { userFormStatus, courseApplied, collegeUrl };

};

const getLeadCourseDetails = async (collegeUrl, courseId) => {

  if(!courseId){
    return null;
  }

  let result = undefined;
  let queryMap = undefined;

  if(collegeUrl){
    queryMap = createCollegeCourseQuery(courseId);
    result = await getCollegesCourseService(queryMap);
  }else{
    queryMap = createUserCourseQuery(courseId);
    result = await getCourseService(queryMap);
  }

  if(result instanceof Error || !result[0]){
    throw new Error("ERROR IN GETTING COURSE DETAILS");
  };

  return result.pop();
 
};

const createCollegeCourseQuery = (courseId) => {
  return (
    {
      attributes: [
        `${C_COURSE}.c_id AS course_id`,
        `${C_COURSE}.course_name`,
        `${C_COURSE}.course_category`,
        `${C_COURSE}.specialization`,
        `${C_COURSE}.course_level`,
        `${C_COURSE}.course_duration`,
        `${C_COURSE}.course_fee`,
        `${C_INSTITUTE}.application_fee`,
        `${C_INSTITUTE}.institute_name`,
        `${C_INSTITUTE}.e_id`,
      ],
      conditions: [`${C_COURSE}.c_id = ?`],
      parameters: [courseId],
      joinTbl: [`LEFT JOIN ${C_INSTITUTE} ON ${C_COURSE}.e_id = ${C_INSTITUTE}.e_id`],
      additional: [`LIMIT 1`],
    }
  )
}

const createUserCourseQuery = (courseId) => {
  return (
    {
      attributes: [
        `${COURSE}.course_id AS course_id`,
        `${COURSE}.course_name`,
        `${COURSE}.course_fullname  AS course_category`,
        `${COURSE}.course_duration AS course_duration`,
      ],
      conditions: [`${COURSE}.course_id = ?`],
      parameters: [courseId],
      additional: [`LIMIT 1`],
    }
  )
};