const { dbQuery } = require("../../../../config/db.config");
const tryCatch = require("../../../../utils/tryCatch");
const {
  errorMsgAuth,
  errorMsgEmpty,
} = require("../../../../utils/errors_message");
const {
  TBL_SCHOOL_SHORTLIST,
  TBL_COMMON_APPLICATION_FORM,
  C_INSTITUTE,
  C_COURSE,
  SCHOOL_DETAIL,
  SCHOOL_FACILITY,
} = require("../../../../models/tables");
const { CLG_STUDENT, SCL_STUDENT } = require("../../../../common/constants");
const { checkCollegeExist, checkSchoolShortlistExist, checkLeadShortlistAlreadyExist } = require("../../../../utils/leadAccessCheck");

const table = TBL_SCHOOL_SHORTLIST;
const tblCommonForm = TBL_COMMON_APPLICATION_FORM;
const college = C_INSTITUTE;
const school = SCHOOL_DETAIL;
const schoolFacility = SCHOOL_FACILITY
const engineering_institute_course = C_COURSE


// @desc Get only required student Shortlist info
// @route GET /api/v1/student/shortlist
// @access Private STUDENTS
const get_studentShortlist = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { us_id, user_type } = req.lead;
  if (!us_id || !user_type) errorMsgAuth("Unauthorized access", res);
  const shortlistAttributes = [`${table}.id`, `${table}.school_id`];
  let attributes = [];
  let tblJoin = "";
  let groupby = "";
  switch (user_type) {
    case CLG_STUDENT:
      attributes = [
        `${college}.institute_name`,
        `${college}.institute_logo`,
        `${tblCommonForm}.form_status`,
        `${tblCommonForm}.id AS common_form_id`,
        `${engineering_institute_course}.course_category`,
        `${engineering_institute_course}.course_name`,
        `${engineering_institute_course}.specialization`,
      ];
      
      tblJoin = `INNER JOIN ${college} ON  ${college}.e_id = ${table}.school_id
                 INNER JOIN ${tblCommonForm} ON ${tblCommonForm}.e_id = ${table}.school_id 
                 INNER JOIN ${engineering_institute_course} ON ${engineering_institute_course}.e_id = ${table}.school_id 
                 AND ${tblCommonForm}.us_id = ${table}.us_id`;
      groupby =`group by school_id`    
      break;
    case SCL_STUDENT:
      attributes = [
        `${school}.school_name`,
        `${school}.school_logo`,
        `${school}.school_type`,
        `${school}.school_affiliated`,
        `${school}.infrastructure`,
      ];
      tblJoin = `LEFT JOIN ${school} ON ${table}.school_id = ${school}.sid`;
      break;
    default:
      errorMsgAuth("Unauthorized access, unknown user", res);
  }

  sql = `SELECT DISTINCT ${[...shortlistAttributes, ...attributes].join(", ")} FROM ${table} ${tblJoin}  WHERE ${table}.us_id = ? AND ${table}.usertype = ? ${groupby}`;
  result = await dbQuery(sql, [us_id, user_type]);
  // console.log(sql, [us_id, user_type])
  if (result.length === 0) errorMsgEmpty("No shortlist found", res);
  if(user_type === SCL_STUDENT){
    for(detail of result){
      const schoolFacilityAttributes = [`${schoolFacility}.cat_name`];
      const schoolInfrastructureQuery = `SELECT ${schoolFacilityAttributes.join(", ")} FROM ${schoolFacility} WHERE FIND_IN_SET(cat_id, '${detail.infrastructure}')`
      const schoolInfrastructureResult = await dbQuery(schoolInfrastructureQuery);
      detail['school_infrastructure'] = schoolInfrastructureResult.map(infra => infra.cat_name) || [];
      delete detail['infrastructure']
    }
  }
  return res.status(200).json({ success: true, data: result });
});

// @desc Post only required student Shortlist info and create entry in common form
// @route POST /api/v1/student/shortlist
// @access Private STUDENTS
const post_studentShortlist = tryCatch(async (req, res) => {

  const { us_id, user_type:type } = req.lead;

  if (!us_id || !type) errorMsgAuth("Unauthorized access", res);

  const { school_id } = req.body;
  if (!school_id) errorMsgEmpty("No school id found", res);

  const leadShortlistDoesNotExist = await checkLeadShortlistAlreadyExist(school_id, us_id, type);
  const shortlistQuery = `INSERT INTO ${table} (us_id, usertype, school_id, date) VALUES (?, ?, ?, NOW())`;

  switch (type) {
    case CLG_STUDENT:
      if(leadShortlistDoesNotExist){ // If Shortlist doesn't exist
        const clgShortlistResult = await dbQuery(shortlistQuery, [us_id, type, school_id]);
        if (!clgShortlistResult.affectedRows) throw new Error("Something went wrong, college not shortlisted");
        const shortlistId = clgShortlistResult.insertId;
        const commonFormQuery = `INSERT INTO ${tblCommonForm} (us_id, e_id, form_status ,date) VALUES (?, ?, ?, NOW())`;
        const commonFormResult = await dbQuery(commonFormQuery, [us_id, school_id, "pending"]);
        let transaction = `DELETE FROM ${table} WHERE id = ?`;
        let transactionResult = undefined;
        if (!commonFormResult.affectedRows) {
          transactionResult = await dbQuery(transaction, [shortlistId]);
        throw new Error("Something went wrong in common form, try shortlisting again");
      }
    }
      break;
      case SCL_STUDENT:
      if(leadShortlistDoesNotExist){ // If shortlist doesn't exist
        const sclShortlistResult = await dbQuery(shortlistQuery, [us_id, type, school_id]);
        if(!sclShortlistResult.affectedRows) throw new Error("Something went wrong, school not shortlisted")
      }
      break;

    default:
      errorMsgAuth("Unauthorized access, unknown user", res);
  }

  return res
    .status(200)
    .json({ success: true, message: "Successfully created Shortlisted" });
});

const delete_studentShortlist = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { us_id, user_type } = req.lead;
  if (!us_id || !user_type) errorMsgAuth("Unauthorized access", res);

  const { id, common_form_id } = req.body;
  if (!common_form_id) throw new Error("No common form id found");
  if (!id) throw new Error("No shortlist id found");

  sql = `DELETE FROM ${table} WHERE id = ? AND us_id = ?`;
  result = await dbQuery(sql, [id, us_id]);

  if (!result.affectedRows)
    throw new Error("Something went wrong, these shortlist may not exist");

  if (user_type === "partner") {
    let sql_ = `DELETE FROM ${tblCommonForm} WHERE id = ? AND us_id = ?`;
    let result_ = await dbQuery(sql_, [common_form_id, us_id]);
    if (!result_.affectedRows)
      throw new Error("Something went wrong in common form");
  }
  return res
    .status(200)
    .json({ success: true, message: "Successfully Deleted Shortlist" });
});

module.exports = {
  get_studentShortlist,
  post_studentShortlist,
  delete_studentShortlist,
};
