const { dbQuery } = require("../../../../config/db.config");
const tryCatch = require("../../../../utils/tryCatch");
const {
  TBL_SCHOOL_STUDENT,
  SCHOOL_DETAIL,
  BUYER_ORDER_DETAILS,
} = require("../../../../models/tables");

const schoolTbl = SCHOOL_DETAIL;
const schoolChildTbl = TBL_SCHOOL_STUDENT;
const buyerTbl = BUYER_ORDER_DETAILS;

const updateChildFormStatus = async (id, status) => {
  try {
    let [sql, result] = [null, null];
    const insertBody = [status, id];
    sql = `UPDATE ${schoolChildTbl} SET form_status = ? WHERE id = ?`;
    result = await dbQuery(sql, insertBody);
    if (!result.affectedRows)
      throw new Error("Error while updating form status");
    return result[0];
  } catch (error) {
    throw new Error(error);
  }
};

const getTotalPartnerSchoolCount = async () => {
  let [sql, result] = [null, null];
  sql = `SELECT COUNT(*) as total_partner_schools FROM ${schoolTbl} WHERE ${schoolTbl}.admission_partner  = ?`;
  result = await dbQuery(sql, ["Y"]);
  return result[0]["total_partner_schools"];
};

const getTotalFormSubmitted = async (id) => {
  let [sql, result] = [null, null];
  const attributes = [
    `${schoolChildTbl}.id`,
    `${schoolChildTbl}.full_name`,
    `${schoolChildTbl}.mother_tongue`,
    `${schoolChildTbl}.gender`,
    `${schoolChildTbl}.dob`,
    `${schoolChildTbl}.religion`,
    `${schoolChildTbl}.category`,
    `${schoolChildTbl}.par_name`,
    `${schoolChildTbl}.par_con`,
    `${schoolChildTbl}.par_email`,
    `${schoolChildTbl}.par_quali`,
    `${schoolChildTbl}.par_occup`,
    `${schoolChildTbl}.par_annual_income`,
    `${schoolChildTbl}.mother_name`,
    `${schoolChildTbl}.mother_con`,
    `${schoolChildTbl}.corr_housne_no`,
    `${schoolChildTbl}.corr_locality`,
    `${schoolChildTbl}.corr_city`,
    `${schoolChildTbl}.corr_state`,
    `${schoolChildTbl}.corr_pin`,
    `${schoolChildTbl}.child_photo`,
    `${schoolChildTbl}.child_birth_certificate`,
    `${schoolChildTbl}.child_address_doc`,
    `${schoolChildTbl}.child_aadhar_card`,
    `${schoolChildTbl}.father_photo`,
    `${schoolChildTbl}.mother_photo`,
    `${schoolChildTbl}.form_status`,
  ];
  sql = `SELECT ${attributes.join(
    ", "
  )} FROM ${schoolChildTbl} WHERE ${schoolChildTbl}.us_id = ?`;
  result = await dbQuery(sql, [id]);
  let totalFormSubmitted = 0;
  let totalFormCompleted = 0;
  result.forEach((item) => {
    if (
      Object.values(item).includes(null) ||
      Object.values(item).includes("")
    ) {
      if (item.form_status !== "pending") {
        updateChildFormStatus(item.id, "pending");
      }
    } else {
      totalFormCompleted++;
      if (item.form_status !== "completed") {
        updateChildFormStatus(item.id, "completed");
      }
    }
    totalFormSubmitted++;
  });
  return { totalFormSubmitted, totalFormCompleted };
};

const getCompletedPaymentCount = async (id) => {
  let [sql, result] = [null, null];
  sql = `SELECT COUNT(*) as total_payment_completed FROM ${buyerTbl} WHERE ${buyerTbl}.BUYER_ID = ?`;
  result = await dbQuery(sql, [id]);
  return result[0][`total_payment_completed`];
};

// @desc Get only required school student Dashboard info
// @route GET /api/v1/student/school-dashboard
// @access Private SCHOOL STUDENT
const getSchoolDashboard = tryCatch(async (req, res) => {
  const { us_id } = req.lead;
  const partnerSchoolCount = await getTotalPartnerSchoolCount();
  const { totalFormSubmitted, totalFormCompleted } =
    await getTotalFormSubmitted(us_id);
  const paymentCompletedCount = await getCompletedPaymentCount(us_id);
  res.status(200).json({
    success: true,
    data: {
      partner_school_count: partnerSchoolCount,
      total_form_submitted: totalFormSubmitted,
      total_form_completed: totalFormCompleted,
      payment_completed_count: paymentCompletedCount,
    },
  });
});

module.exports = {
  getSchoolDashboard,
};
