const { dbQuery } = require("../../../../config/db.config");
const tryCatch = require("../../../../utils/tryCatch");
const {
  errorMsgAuth,
  errorMsgEmpty,
} = require("../../../../utils/errors_message");
const updateColumn = require("../../../../utils/updateColumn");
const { TBL_SCHOOL_STUDENT } = require("../../../../models/tables");
const {filterRequiredFields, exportFieldsName} = require("../../requiredFieldsForUpdate.util");

const table = TBL_SCHOOL_STUDENT;

// @desc all the added school child
// @route GET /api/v1/student/manage-child
// @access Private STUDENTS
const get_manageChild = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { us_id } = req.lead;
  if (!us_id) errorMsgAuth("Unauthorized access", res);
  const attributes = [
    `${table}.id`,
    `${table}.full_name`,
    `${table}.gender`,
    `${table}.dob`,
    `${table}.std`,
    `${table}.form_status`,
  ];
  const insertBody = [us_id];
  sql = `SELECT ${attributes.join(", ")}  FROM ${table} WHERE us_id = ?`;
  result = await dbQuery(sql, insertBody);
  if (result.length === 0) errorMsgEmpty("No school student found", res);
  const total = result.length;
  return res.status(200).json({
    success: true,
    data: result,
    records: total,
  });
});

// @desc add school child
// @route POST /api/v1/student/manage-child
// @access Private STUDENTS
const post_manageChild = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { us_id } = req.lead;
  if (!us_id) errorMsgAuth("Unauthorized access", res);
  const {
    full_name,
    gender,
    std, // class to admit
    dob,
  } = req.body;
  if (!full_name || !gender || !std || !dob)
    throw new Error("Please fill all the fields");

  const attributes = [
    `${table}.us_id`,
    `${table}.full_name`,
    `${table}.gender`,
    `${table}.std`,
    `${table}.dob`,
  ];
  const insertBody = [
    us_id,
    full_name,
    gender,
    std,
    dob,
  ]
  sql = `INSERT INTO ${table} (${attributes.join(", ")}) VALUES (?, ?, ?, ?, ?)`;
  result = await dbQuery(sql, insertBody);
  if (!result.affectedRows) throw new Error("No child added not added");
  return res.status(200).json({
    success: true,
    message: "Child added",
  });
});

// @desc update school child
// @route PUT /api/v1/student/manage-child
// @access Private STUDENTS
const update_manageChild = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const [{ us_id }, { id, update }] = [req.lead, req.body];
  if (!us_id) errorMsgAuth("Unauthorized access", res);
  if (!id || !update) throw new Error("Please fill all the fields");

  const { fieldsForManagingChild } = exportFieldsName;
  const requiredFieldsObj = filterRequiredFields(update, fieldsForManagingChild);
  if (Object.keys(requiredFieldsObj).length === 0) throw new Error("No required field to update");
  const { column, value } = updateColumn(requiredFieldsObj);

  sql = `UPDATE ${table} SET ${column} WHERE id = ? AND us_id = ?`;
  result = await dbQuery(sql, [...value, id, us_id]);
  if (!result.affectedRows)
    throw new Error("No child updated, does child exist ?");
  return res.status(200).json({
    success: true,
    message: "Child updated",
  });
});

// @desc delete school child
// @route DELETE /api/v1/student/manage-child
// @access Private STUDENTS
const delete_manageChild = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const [{ us_id }, { id }] = [req.lead, req.body];
  if (!us_id) errorMsgAuth("Unauthorized access", res);
  if (!id) throw new Error("Please fill all the fields");
  sql = `DELETE FROM ${table} WHERE id = ? AND us_id = ?`;
  result = await dbQuery(sql, [id, us_id]);
  if (!result.affectedRows)
    throw new Error("No child deleted, does child exist ?");
  return res.status(200).json({
    success: true,
    message: "Child deleted",
  });
});

module.exports = {
  get_manageChild,
  post_manageChild,
  update_manageChild,
  delete_manageChild,
};
