const { dbQuery } = require("../../../../config/db.config");
const tryCatch = require("../../../../utils/tryCatch");
const {
  errorMsgAuth,
  errorMsgEmpty,
} = require("../../../../utils/errors_message");
const {
  getSchoolsBasicDetails,
  checkSchoolExist,
  checkSchoolAlreadyAddedInCart,
} = require("../../../../utils/leadAccessCheck");
const { TBL_STUDENT_CART, SCHOOL_DETAIL, TBL_SCHOOL_STUDENT } = require("../../../../models/tables");
const { SCHOOL } = require("../../../../common/constants");

const table = TBL_STUDENT_CART;
const schoolTbl = SCHOOL_DETAIL;
const childTbl = TBL_SCHOOL_STUDENT;

//@desc get all applied schools in cart
//@route GET /api/v1/student/my-cart
//@access Private STUDENTS
const getSchoolFromCart = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { us_id } = req.lead;
  let joinTbl = `LEFT JOIN ${schoolTbl} ON ${schoolTbl}.sid = ${table}.institute_id LEFT JOIN ${childTbl} ON ${childTbl}.id = ${table}.child_id`;
  const attributes = [
    `${table}.id`,
    `${table}.institute_name`,
    `${table}.std`,
    `${table}.session`,
    `${table}.reg_fee`,
    `${childTbl}.full_name`,
    `${schoolTbl}.school_logo`,
  ]
  const insertBody = [us_id];
  sql = `SELECT ${attributes.join(", ")} FROM ${table} ${joinTbl} WHERE ${table}.us_id = ?`;
  result = await dbQuery(sql, insertBody);
  if(!result.length) errorMsgEmpty("Cart is empty", res);
  let  getFeesTotal = result.reduce((acc, curr) => {
    return acc + Number(curr.reg_fee);
  }, 0);
  return res.status(200).json({
    success: true,
    data: result,
    total_applications_fees: getFeesTotal,
  });
});

//@desc remove school from cart
//@route DELETE /api/v1/student/my-cart
//@access Private STUDENTS
const deleteSchoolFromCart = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { us_id } = req.lead;
  const { id } = req.body;
  if (!id) throw new Error("Some required Fields are missing");
  const doesSchoolExistInCart = await checkSchoolAlreadyAddedInCart(us_id, null, null, id);
  if (!doesSchoolExistInCart) throw new Error("School not in cart or might not exist");
  sql = `DELETE FROM ${table} WHERE id = ? AND us_id = ?`;
  result = await dbQuery(sql, [id, us_id]);
  if (!result.affectedRows) throw new Error("School not removed from cart");
  return res.status(200).json({
    success: true,
    message: "School removed from cart",
  });
});

//@desc apply for school
//@route POST /api/v1/student/my-cart
//@access Private STUDENTS
const addSchoolToCart = tryCatch(async (req, res) => {
  let [sql, result] = [null, null];
  const { us_id } = req.lead;
  const { school_id, child_id } = req.body;
  if ((!school_id, !child_id ))
    throw new Error("Some required Fields are missing");

  const isPartner = "Y";
  const doesSchoolExist = await checkSchoolExist(school_id, isPartner);
  const schoolDetailsIsPartner = await getSchoolsBasicDetails(
    school_id,
    isPartner
  );
  if (!doesSchoolExist || !schoolDetailsIsPartner)
    throw new Error("This school might not exist or is not a partner");

  const checkSchoolAlreadyAdded = await checkSchoolAlreadyAddedInCart(
    us_id,
    school_id,
    child_id
  );
  if (!!checkSchoolAlreadyAdded) {
    throw new Error("School already in the cart");
  }

  const childBasicDetails = await getChildBasicDetails(child_id);
  if(childBasicDetails.form_status !== "completed") throw new Error("Child's form is not complete, Complete the form first");
  
  const date = new Date();
  const session = date.getFullYear() + " - " + (date.getFullYear() + 1);
  const schoolName = schoolDetailsIsPartner.school_name;
  const schoolRegFee = schoolDetailsIsPartner.asc_reg_fee || 0;
  const attributes = [
    `${table}.institute_name`,
    `${table}.institute_id`,
    `${table}.institute_type`,
    `${table}.us_id`,
    `${table}.child_id`,
    `${table}.std`,
    `${table}.session`,
    `${table}.reg_fee`,
    `${table}.date`,
  ];
  const insertBody = [
    schoolName,
    school_id,
    SCHOOL,
    us_id,
    child_id,
    childBasicDetails.std,
    session,
    schoolRegFee,
  ]
  sql = `INSERT INTO ${table} (${attributes.join(", ")}) VALUES(?,?,?,?,?,?,?,?,NOW())`;
  result = await dbQuery(sql, insertBody);

  if (!result.affectedRows) throw new Error("School not added to cart");

  return res.status(200).json({
    success: true,
    message: "School added to cart",
  });
});

const getChildBasicDetails = async (child_id) => {
  try {
    let [sql, result] = [null, null];
    const attributes = [
      `${childTbl}.full_name`,
      `${childTbl}.std`,
      `${childTbl}.form_status`,
    ]
    const insertBody = [child_id];
    sql = `SELECT ${attributes.join(", ")} FROM ${childTbl} WHERE ${childTbl}.id = ?`;
    result = await dbQuery(sql, insertBody);
    if (!result.length) throw new Error("Child not found");
    return result[0];
  } catch (error) {
    throw new Error(error.message);
  }
};

module.exports = {
  getSchoolFromCart,
  deleteSchoolFromCart,
  addSchoolToCart,
};