const { dbQuery } = require("../../../../config/db.config");
const {
  C_ADDITIONAL,
  C_COURSE,
  C_INSTITUTE,
  C_MEDIA,
  C_CONTACT,
  TBL_AWARDS,
  TBL_ACCORDION_CONTENT,
  SC_FACILITIES,
  COMPANY_PLACEMENT,
  STATE_TBL,
  CITY_TBL,
  COURSE,
  CATEGORY,
  SPECIALIZATION,
  COURSE_DELIVERY,
  COURSELEVEL,
} = require("../../../../models/tables");
const tryCatch = require("../../../../utils/tryCatch");
const { getCollegeIdBasedOnUrl } = require("../../../../utils/leadAccessCheck");
const {
  getCollegeService,
  getCollegeAwardService,
  getCollegeFaqService,
  getCollegesCourseService,
} = require("../../../../services/college.service");
const {
  getCompanyPlacementPaginationService,
} = require("../../../../services/master/companyPlacement.service");

// @desc get details on one college based on id, its for public to view
// @route GET /api/v1/public/college/?url=college-url
// @access PUBLIC
const getCollegeDetailsForPublic = tryCatch(async (req, res) => {
  const { url } = req.query;

  // get college id
  const e_id = await getCollegeIdBasedOnUrl(url);
  if (!e_id) {
    return res.status(404).json({
      status: "fail",
      message: "College not found",
      data: [],
    });
  }

  const queryMapCollege = {
    attributes: setCollegeAttributes(),
    conditions: [`${C_INSTITUTE}.e_id = ?`],
    parameters: [e_id],
    additional: [`LIMIT 1`],
    joinTbl: [
      `LEFT JOIN ${C_MEDIA} ON ${C_INSTITUTE}.e_id = ${C_MEDIA}.e_id`,
      `LEFT JOIN ${C_ADDITIONAL} ON ${C_INSTITUTE}.e_id = ${C_ADDITIONAL}.e_id`,
      `LEFT JOIN ${C_CONTACT} ON ${C_INSTITUTE}.e_id = ${C_CONTACT}.e_id`,
      `LEFT JOIN ${STATE_TBL} ON (${STATE_TBL}.sid = ${C_CONTACT}.state OR ${STATE_TBL}.state = ${C_CONTACT}.state)`,
      `LEFT JOIN ${CITY_TBL} ON (${CITY_TBL}.ctid = ${C_CONTACT}.city OR ${CITY_TBL}.cname = ${C_CONTACT}.city)`
    ],
  };
  const clgDetailsResult = await getCollegeService(queryMapCollege);
  if (clgDetailsResult instanceof Error || !clgDetailsResult.length) {
    res.statusCode = 404;
    throw new Error("College not found");
  }
  const collegeDetails = clgDetailsResult.pop();

  // SET COLLEGE DETAILS FROM OTHER TABLES
  const infrastructureDetails = await getInfrastructureDetails(collegeDetails.facilities);
  const placementCompany = await getPlacementCompany(collegeDetails.placement_company_list);
  const { courses, pgCourses } = await setCollegeCourses(e_id);

  collegeDetails.infrastructure = infrastructureDetails;
  collegeDetails.placement_company_list = placementCompany;
  collegeDetails.courses = courses;
  collegeDetails.pg_courses = pgCourses;
  collegeDetails.awards = await setCollegeAwards(e_id) || [];
  collegeDetails.faq = await setCollegeFaq(e_id) || [];

  return res.status(200).json({
    success: true,
    data: collegeDetails,
  });

});

const getInfrastructureDetails = async (facilities) => {
  if (!facilities) return [];
  try {
    let [sql, result] = [null, null];
    const attributes = [`${SC_FACILITIES}.name`, `${SC_FACILITIES}.nr_image`];
    sql = `SELECT ${attributes.join(
      ", "
    )} FROM ${SC_FACILITIES} WHERE id IN (${facilities})`;
    result = await dbQuery(sql);
    return result;
  } catch (error) {
    throw new Error(error);
  }
};

const getPlacementCompany = async (placementCompanyList) => {
  if (!placementCompanyList) return [];
  try {
    const queryMap = {
      attributes: [
        `${COMPANY_PLACEMENT}.company_name`,
        `${COMPANY_PLACEMENT}.company_logo`,
      ],
      conditions: [`${COMPANY_PLACEMENT}.com_id IN (${placementCompanyList})`, `${COMPANY_PLACEMENT}.active = 'Y'`],
    };
    const { result } = await getCompanyPlacementPaginationService(queryMap);
    return result;
  } catch (error) {
    throw new Error(error);
  }
};

// @desc get recommended colleges and also limit the number of colleges
// @route GET /api/v1/public/recommended-colleges
// @access PUBLIC
const getRecommendedColleges = tryCatch(async (req, res) => {
  const { limit = 7 } = req.query;

  const queryMap = {
    attributes: [
      `${C_INSTITUTE}.e_id`,
      `${C_INSTITUTE}.institute_name AS name`,
      `${C_INSTITUTE}.institute_short_name`,
      `${C_INSTITUTE}.institute_logo AS logo`,
      `${C_INSTITUTE}.institute_url AS url`,
      `${STATE_TBL}.state AS location`,
    ],
    conditions: [
      `${C_INSTITUTE}.is_partner = 1`,
      `${C_INSTITUTE}.active = 'Y'`,
      `${C_INSTITUTE}.top_college = 'Y'`,
    ],
    additional: [
      `ORDER BY ${C_INSTITUTE}.priority ASC`,
      `LIMIT ${Number(limit) || 5}`,
    ],
    joinTbl: [
      `LEFT JOIN ${C_CONTACT} ON ${C_CONTACT}.e_id = ${C_INSTITUTE}.e_id`,
      `LEFT JOIN ${STATE_TBL} ON (${STATE_TBL}.sid = ${C_CONTACT}.state OR ${STATE_TBL}.state = ${C_CONTACT}.state)`,
    ],
  };

  const result = await getCollegeService(queryMap);
  if (result instanceof Error) {
    res.statusCode = 404;
    throw result;
  }

  return res.status(200).json({
    success: true,
    data: result,
  });
});

module.exports = {
  getCollegeDetailsForPublic,
  getInfrastructureDetails,
  getRecommendedColleges,
};

const setCollegeAttributes = () => {
  const collegeAttributesArr = [];
  const metaAttributes = [
    `${C_INSTITUTE}.pageTitle`,
    `${C_INSTITUTE}.metaDescription`,
    `${C_INSTITUTE}.metaKeywords`,
  ];
  collegeAttributesArr.push(...metaAttributes);
  const collegeAttributes = [
    `${C_INSTITUTE}.e_id`,
    `${C_INSTITUTE}.institute_name AS name`,
    `${C_INSTITUTE}.institute_short_name`,
    `${C_INSTITUTE}.college_type AS type`,
    `${C_INSTITUTE}.approved_by`,
    `${C_INSTITUTE}.establish`,
    `${C_INSTITUTE}.institute_logo AS logo`,
    `${C_INSTITUTE}.small_college_logo`,
    `${C_INSTITUTE}.is_partner`,
    `${C_INSTITUTE}.institute_url AS url`,
    // Hostel
    `${C_INSTITUTE}.boy_h_1 AS boy_single_seater`,
    `${C_INSTITUTE}.boy_h_2 AS boy_double_seater`,
    `${C_INSTITUTE}.boy_h_3 AS boy_triple_seater`,
    `${C_INSTITUTE}.girl_h_1 AS girl_single_seater`,
    `${C_INSTITUTE}.girl_h_2 AS girl_double_seater`,
    `${C_INSTITUTE}.girl_h_3 AS girl_triple_seater`,
    //Placement
    `${C_INSTITUTE}.placement_company_list`,
    //Facilities
    `${C_INSTITUTE}.facilities`,
  ];
  collegeAttributesArr.push(...collegeAttributes);

  const collegeMedia = [
    `${C_MEDIA}.image1`,
    `${C_MEDIA}.image2`,
    `${C_MEDIA}.image3`,
    `${C_MEDIA}.image4`,
    `${C_MEDIA}.image5`,
    `${C_MEDIA}.image6`,
    `${C_MEDIA}.video1`,
    `${C_MEDIA}.video2`,
    `${C_MEDIA}.video3`,
    `${C_MEDIA}.video4`,
  ];
  collegeAttributesArr.push(...collegeMedia);

  const collegeAbout = [
    `${C_ADDITIONAL}.about_institute AS about`,
    `${C_ADDITIONAL}.ranking`,
  ];
  collegeAttributesArr.push(...collegeAbout);

  const collegeLocationAttributes = [
    `${CITY_TBL}.cname AS city`,
    `${STATE_TBL}.state AS state`,
  ];
  collegeAttributesArr.push(...collegeLocationAttributes);
  return collegeAttributesArr;
};

const setCollegeAwards = async (e_id) => {
  try {
    const queryMapCollegeAwards = {
      attributes: [`${TBL_AWARDS}.awd_title`, `${TBL_AWARDS}.awd_descriptipn`],
      conditions: [`${TBL_AWARDS}.e_id = ?`, `${TBL_AWARDS}.is_active = 'Y'`],
      parameters: [e_id],
    };
    const clgAwardsResult = await getCollegeAwardService(queryMapCollegeAwards);
    console.log(clgAwardsResult);
    return clgAwardsResult;
  } catch (error) {
    throw new Error(error);
  }
};

const setCollegeFaq = async (e_id) => {
  try {
    const queryMapCollegeFaq = {
      attributes: [
        `${TBL_ACCORDION_CONTENT}.question`,
        `${TBL_ACCORDION_CONTENT}.answer`,
      ],
      conditions: [`${TBL_ACCORDION_CONTENT}.org_id = ?`, `${TBL_ACCORDION_CONTENT}.active = 'Y'`],
      parameters: [e_id],
    };
    const clgFaqResult = await getCollegeFaqService(queryMapCollegeFaq);
    return clgFaqResult;
  } catch (error) {
    throw new Error(error);
  }
};

const setCollegeCourses = async (e_id) => {
  try {
    const queryMapCollegeCourse = {
      attributes: [
        `${C_COURSE}.c_id`,
        `${CATEGORY}.cat_name AS course_category`,
        `${COURSE}.course_name AS course_name`,
        `${SPECIALIZATION}.spce_name AS specialization`,
        `${C_COURSE}.course_duration`,
        `${C_COURSE}.course_duration_type`,
        `${COURSE_DELIVERY}.name AS mode_learning`,
        `${COURSELEVEL}.name AS course_level`,
        `${C_COURSE}.course_fee`,
      ],
      conditions: [`${C_COURSE}.e_id = ?`],
      parameters: [e_id],
      joinTbl: [
        `LEFT JOIN ${COURSE} ON (${COURSE}.course_id = ${C_COURSE}.course_name OR ${COURSE}.course_name = ${C_COURSE}.course_name)`,
        `LEFT JOIN ${CATEGORY} ON (${CATEGORY}.cat_id = ${C_COURSE}.course_category OR ${CATEGORY}.cat_name = ${C_COURSE}.course_category)`,
        `LEFT JOIN ${SPECIALIZATION} ON (${SPECIALIZATION}.spec_id = ${C_COURSE}.specialization OR ${SPECIALIZATION}.spce_name = ${C_COURSE}.specialization)`,
        `LEFT JOIN ${COURSE_DELIVERY} ON (${COURSE_DELIVERY}.id = ${C_COURSE}.mode_learning OR ${COURSE_DELIVERY}.name = ${C_COURSE}.mode_learning)`,
        `LEFT JOIN ${COURSELEVEL} ON (${COURSELEVEL}.id = ${C_COURSE}.course_level OR ${COURSELEVEL}.name = ${C_COURSE}.course_level)`
      ],
      additional: [`GROUP BY ${C_COURSE}.c_id`]
    };
    const clgCoursesResult = await getCollegesCourseService(
      queryMapCollegeCourse
    );
    const courses = [] , pgCourses = [];
    clgCoursesResult.forEach((course) => {
      if (course.course_level === "PG Courses") {
        pgCourses.push(course);
      } else {
        courses.push(course);
      }
    });
    return { courses, pgCourses };
  } catch (error) {
    throw new Error(error);
  }
};
